package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.model.BlueprintIdBundle;
import com.atlassian.confluence.plugins.createcontent.services.BlueprintResolver;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.web.WebInterfaceManager;
import com.atlassian.plugin.web.descriptors.WebItemModuleDescriptor;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.BLUEPRINT_PARAM_KEY;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.CREATE_DIALOG_CONTENT_SECTION;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

public class DefaultBlueprintResolver implements BlueprintResolver {
    private static final Logger log = LoggerFactory.getLogger(DefaultBlueprintResolver.class);

    private final WebInterfaceManager webInterfaceManager;
    private final ContentBlueprintManager contentBlueprintManager;

    public DefaultBlueprintResolver(WebInterfaceManager webInterfaceManager,
                                    ContentBlueprintManager contentBlueprintManager) {
        this.webInterfaceManager = webInterfaceManager;
        this.contentBlueprintManager = contentBlueprintManager;
    }

    @Override
    public WebItemModuleDescriptor getWebItemMatchingBlueprint(UUID blueprintId) {
        String moduleCompleteKey = contentBlueprintManager.getById(blueprintId).getModuleCompleteKey();
        return getWebItemMatchingBlueprint(moduleCompleteKey);
    }

    @Override
    public WebItemModuleDescriptor getWebItemMatchingBlueprint(final String blueprintModuleKey) {
        List<WebItemModuleDescriptor> items = webInterfaceManager.getItems(CREATE_DIALOG_CONTENT_SECTION);
        for (WebItemModuleDescriptor webItem : items) {
            String moduleKey = webItem.getParams().get(BLUEPRINT_PARAM_KEY);
            if (StringUtils.isBlank(moduleKey)) {
                // no blueprint for this item - can't match.
                continue;
            }
            String pluginKey = webItem.getPluginKey();
            final String blueprintKey = getModuleCompleteKeyFromRelative(pluginKey, moduleKey).getCompleteKey();
            if (blueprintModuleKey.equals(blueprintKey)) {
                return webItem;
            }
        }
        return null;
    }

    @Override
    public ContentTemplateRef resolveTemplateRef(ContentTemplateRef templateRef) {
        if (templateRef.getTemplateId() > 0)
            return templateRef; // this template points to a custom PageTemplate in the DB - use that

        ContentBlueprint globalBlueprint;
        ContentBlueprint parentBlueprint = templateRef.getParent();
        if (isNotBlank(parentBlueprint.getSpaceKey())) {
            // A space-level custom Blueprint. Look for the global version of this Blueprint.
            ModuleCompleteKey moduleCompleteKey = new ModuleCompleteKey(parentBlueprint.getModuleCompleteKey());
            globalBlueprint = contentBlueprintManager.getPluginBackedContentBlueprint(moduleCompleteKey, (String) null);
        } else {
            // Already at global level. This one's template ref will do.
            globalBlueprint = parentBlueprint;
        }

        List<ContentTemplateRef> contentTemplateRefs = Lists.newArrayList();
        if (globalBlueprint.getContentTemplateRefs() != null)
            contentTemplateRefs.addAll(globalBlueprint.getContentTemplateRefs());
        if (globalBlueprint.getIndexPageTemplateRef() != null)
            contentTemplateRefs.add(globalBlueprint.getIndexPageTemplateRef());

        return findTemplateWithKey(templateRef.getModuleCompleteKey(), contentTemplateRefs);
    }

    @Nonnull
    @Override
    public ContentBlueprint resolveContentBlueprint(@Nonnull String blueprintId, @Nullable String spaceKey) throws IllegalArgumentException {
        ContentBlueprint blueprint = contentBlueprintManager.getById(UUID.fromString(blueprintId));
        return getContentBlueprint(blueprintId, blueprint.getModuleCompleteKey(), spaceKey);
    }

    @Override
    @Nonnull
    public ContentBlueprint getContentBlueprint(String contentBlueprintId, String blueprintModuleCompleteKey, String spaceKey) throws IllegalArgumentException {
        UUID blueprintUUID = null;
        if (isNotBlank(contentBlueprintId)) {
            blueprintUUID = UUID.fromString(contentBlueprintId);
        }
        ModuleCompleteKey moduleCompleteKey = null;
        if (isNotBlank(blueprintModuleCompleteKey)) {
            moduleCompleteKey = new ModuleCompleteKey(blueprintModuleCompleteKey);
        }
        if (blueprintUUID == null && moduleCompleteKey == null)
            throw new IllegalArgumentException("Not enough data to get Blueprint: " + contentBlueprintId + ", " + blueprintModuleCompleteKey);

        BlueprintIdBundle idBundle = new BlueprintIdBundle(blueprintUUID, moduleCompleteKey, spaceKey);
        ContentBlueprint blueprint = get(idBundle);
        if (blueprint == null)
            throw new IllegalArgumentException("Unknown Blueprint: " + idBundle);

        return blueprint;
    }

    private ContentBlueprint get(BlueprintIdBundle idBundle) {
        UUID incomingId = idBundle.getBlueprintId();
        ModuleCompleteKey moduleKey = idBundle.getBlueprintModuleKey();
        if (moduleKey == null) {
            return contentBlueprintManager.getById(incomingId);
        }

        ContentBlueprint bp = contentBlueprintManager.getPluginBackedContentBlueprint(moduleKey, idBundle.getSpaceKey());
        if (bp == null) {
            log.warn("No ContentBlueprint found for id-bundle: " + idBundle);
            return null;
        }
        UUID foundId = bp.getId();
        if (incomingId != null && !foundId.equals(incomingId)) {
            // Not the end of the world but not expected. In future, getPluginBackedContentBlueprint should allow for
            // retrieval of multiple CBPs with the same key and space, and we can confirm here that one of those CBPs
            // matches the incoming UUID.
            log.warn("ContentBlueprint id mismatch for id-bundle: " + idBundle + ". Found blueprint with id: " + foundId);
        }
        return bp;
    }

    private ContentTemplateRef findTemplateWithKey(String moduleCompleteKey,
                                                   List<ContentTemplateRef> contentTemplateRefs) {
        for (ContentTemplateRef contentTemplateRef : contentTemplateRefs) {
            if (contentTemplateRef.getModuleCompleteKey().equals(moduleCompleteKey))
                return contentTemplateRef;
        }

        throw new IllegalStateException("No content template ref found with module key: " + moduleCompleteKey);
    }

    /**
     * Return a complete module key given the supplied plugin key and module key. The trick here is that the module
     * key may be relative (i.e. a normal module key) or a complete key itself, in which case it should be returned as
     * is.
     *
     * @param pluginKey a fall-back plugin key in case the module key is actually "relative"
     * @param moduleKey a plugin module key that be relative or complete
     * @return a complete key
     */
    private ModuleCompleteKey getModuleCompleteKeyFromRelative(String pluginKey, String moduleKey) {
        try {
            return new ModuleCompleteKey(moduleKey);
        } catch (Exception e) {
            return new ModuleCompleteKey(pluginKey, moduleKey);
        }
    }
}
