package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.confluence.plugins.createcontent.AoBackedManager;
import com.atlassian.confluence.plugins.createcontent.activeobjects.PluginBackedBlueprintAo;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.sal.api.transaction.TransactionCallback;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;
import java.util.UUID;

import static com.google.common.collect.Lists.newArrayList;

public abstract class AbstractAoManager<O extends PluginBackedBlueprint, A extends PluginBackedBlueprintAo> implements AoBackedManager<O, A> {
    private final Class<A> aoClass;

    protected final ActiveObjects activeObjects;
    protected final HelperAoManager<A> helperAoManager;
    private final HelperAoManager.HelperCallback<A> helperCallback = new HelperAoManager.HelperCallback<A>() {
        @Override
        public void onDelete(final A ao) {
            internalDeleteAo(ao);
        }

        @Override
        public void onDeleteAll(final A[] aos) {
            internalDeleteAllAo(aos);
        }
    };

    protected AbstractAoManager(@Nonnull ActiveObjects activeObjects, @Nonnull Class<A> aoClass) {
        this.activeObjects = activeObjects;
        this.aoClass = aoClass;
        helperAoManager = new HelperAoManager<A>(activeObjects, aoClass);
    }

    @Nullable
    @Override
    public O getById(@Nonnull final UUID id) {
        A ao = getAoById(id);
        return ao != null ? build(ao) : null;
    }

    @Nullable
    @Override
    public A getAoById(@Nonnull final UUID id) {
        return activeObjects.executeInTransaction(new TransactionCallback<A>() {
            @Override
            public A doInTransaction() {
                return internalGetAoById(id);
            }
        });
    }

    @Nullable
    @Override
    public O getCloneByModuleCompleteKey(@Nonnull final ModuleCompleteKey moduleCompleteKey) {
        A[] ao = getAosByModuleCompleteKey(moduleCompleteKey, true);
        return ao.length > 0 ? build(ao[0]) : null;
    }

    @Nonnull
    @Override
    public List<O> getNonClonesByModuleCompleteKey(@Nonnull final ModuleCompleteKey moduleCompleteKey) {
        return activeObjects.executeInTransaction(new TransactionCallback<List<O>>() {
            @Override
            public List<O> doInTransaction() {
                List<O> result = newArrayList();
                A[] aos = internalGetAosByModuleCompleteKey(moduleCompleteKey, false);
                for (A ao : aos) {
                    result.add(build(ao));
                }
                return result;
            }
        });
    }

    /**
     * Find one object (if pluginClone == true) / multiple objects (if pluginclone == false) by its module complete key
     */
    @Nonnull
    protected A[] getAosByModuleCompleteKey(@Nonnull final ModuleCompleteKey moduleCompleteKey, final boolean pluginClone) {
        return activeObjects.executeInTransaction(new TransactionCallback<A[]>() {
            @Override
            public A[] doInTransaction() {
                return internalGetAosByModuleCompleteKey(moduleCompleteKey, pluginClone);
            }
        });
    }

    @Nonnull
    @Override
    public List<O> getAll() {
        List<O> result = newArrayList();

        A[] aos = activeObjects.find(aoClass);
        if (aos != null && aos.length != 0) {
            for (A ao : aos) {
                result.add(build(ao));
            }
        }

        return result;
    }

    @Nonnull
    protected List<O> getAll(String query, Object... args) {
        List<O> result = newArrayList();

        A[] aos = activeObjects.find(aoClass, query, args);
        if (aos != null && aos.length != 0) {
            for (A ao : aos) {
                result.add(build(ao));
            }
        }

        return result;
    }

    @Nonnull
    @Override
    public O create(@Nonnull final O original) {
        return build(createAo(original));
    }

    @Nonnull
    @Override
    public A createAo(@Nonnull final O original) {
        return activeObjects.executeInTransaction(new TransactionCallback<A>() {
            @Override
            public A doInTransaction() {
                return internalCreateAo(original);
            }
        });
    }

    @Nonnull
    @Override
    public final O update(@Nonnull final O object) {
        activeObjects.executeInTransaction(new TransactionCallback<Void>() {
            @Override
            public Void doInTransaction() {
                internalUpdateAo(object);
                return null;
            }
        });
        return object;
    }

    @Nonnull
    @Override
    public final A updateAo(@Nonnull final O object) {
        return activeObjects.executeInTransaction(new TransactionCallback<A>() {
            @Override
            public A doInTransaction() {
                return internalUpdateAo(object);
            }
        });
    }

    @Override
    public final boolean delete(@Nonnull UUID id) {
        return helperAoManager.delete(id, helperCallback);
    }

    @Override
    public final void delete(@Nonnull final A object) {
        helperAoManager.delete(object, helperCallback);
    }

    @Override
    public final int deleteAll() {
        return helperAoManager.deleteAll(helperCallback);
    }

    @Nonnull
    protected A[] internalGetAosByModuleCompleteKey(@Nonnull final ModuleCompleteKey moduleCompleteKey, final boolean pluginClone) {
        return activeObjects.find(aoClass, "PLUGIN_MODULE_KEY = ? AND PLUGIN_CLONE = ?", moduleCompleteKey.getCompleteKey(), pluginClone);
    }

    @Nullable
    protected A internalGetAoById(@Nonnull UUID id) {
        return helperAoManager.internalGetAoById(id);
    }

    @Nonnull
    protected abstract A internalCreateAo(@Nonnull O original);

    @Nonnull
    protected abstract A internalUpdateAo(@Nonnull O object);

    protected abstract void internalDeleteAo(@Nonnull A ao);

    protected void internalDeleteAllAo(@Nonnull A[] aos) {
        for (A ao : aos) {
            internalDeleteAo(ao);
        }
    }

    @Nonnull
    protected abstract O build(@Nonnull A ao);
}
