package com.atlassian.confluence.plugins.createcontent.factory;

import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.Expansions;
import com.atlassian.confluence.api.model.content.ContentBody;
import com.atlassian.confluence.api.model.content.ContentRepresentation;
import com.atlassian.confluence.api.model.content.Space;
import com.atlassian.confluence.api.model.content.template.ContentTemplate;
import com.atlassian.confluence.api.model.content.template.ContentTemplateId;
import com.atlassian.confluence.api.model.content.template.ContentTemplateType;
import com.atlassian.confluence.api.model.plugin.ModuleCompleteKey;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.fugue.Option;
import com.google.common.base.Strings;

import java.util.stream.Collectors;

/**
 * Factory builds ContentTemplate from PageTemplate.
 */
public class ContentTemplateFactory {
    private static ContentTemplate buildFrom(Option<Space> space,
                                             ContentTemplateRef templateRef, PageTemplate pageTemplate, Expansion[] expansions,
                                             boolean newlyCreatedPageTemplate) {
        Expansions myExpansions = new Expansions(expansions);

        ContentTemplate.ContentTemplateBuilder contentTemplateBuilder = ContentTemplate.builder()
                .templateType(ContentTemplateType.BLUEPRINT)
                .originalTemplate(new ModuleCompleteKey(templateRef.getModuleCompleteKey()))
                .referencingBlueprint(templateRef.getParent().getContentBlueprintId())
                .name(pageTemplate.getName())
                .description(pageTemplate.getDescription())
                .labels(
                        pageTemplate.getLabels()
                                .stream()
                                .map(input -> LabelFactory.buildFrom(input, myExpansions))
                                .collect(Collectors.toList()));

        if (space.isDefined()) {
            contentTemplateBuilder.space(space.get());
        }

        if (templateRef.getTemplateId() != 0 || newlyCreatedPageTemplate) {
            contentTemplateBuilder.templateId(ContentTemplateId.fromString(String.valueOf(pageTemplate.getId())));
        } else //When template is never modified, we use uuid to identify it.
        {
            contentTemplateBuilder.templateId(ContentTemplateId.fromString(templateRef.getId().toString()));
        }

        if (myExpansions.canExpand(ContentTemplate.Expansions.BODY) && !Strings.isNullOrEmpty
                (pageTemplate.getContent())) {
            ContentBody contentBody = ContentBody.contentBodyBuilder()
                    .value(pageTemplate.getContent())
                    .representation(ContentRepresentation.STORAGE)
                    .build();
            contentTemplateBuilder.body(contentBody);
        }

        return contentTemplateBuilder.build();
    }

    public static ContentTemplate buildFromNewPageTemplate(Option<Space> space,
                                                           ContentTemplateRef templateRef, PageTemplate pageTemplate, Expansion[] expansions) {
        return buildFrom(space, templateRef, pageTemplate, expansions, true);
    }

    public static ContentTemplate buildFrom(Option<Space> space,
                                            ContentTemplateRef templateRef, PageTemplate pageTemplate, Expansion[] expansions) {
        return buildFrom(space, templateRef, pageTemplate, expansions, false);
    }
}
