package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.confluence.api.service.accessmode.AccessModeService;
import com.atlassian.confluence.api.service.accessmode.ReadOnlyAccessAllowed;
import com.atlassian.confluence.plugins.createcontent.BlueprintStateController;
import com.atlassian.confluence.plugins.createcontent.SpaceBlueprintStateController;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import org.apache.commons.lang3.StringUtils;

import javax.ws.rs.DELETE;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import java.util.UUID;

@Path("/modules")
public class BlueprintStateResource extends AbstractRestResource {
    public static final String NON_SPACE_ADMIN_PERMISSION_DENIED_RESPONSE =
            "Only space administrators can enable / disable plugin modules per space";

    public static final String PARAM_SPACE_KEY = "spaceKey";
    public static final String PARAM_BLUEPRINT_ID = "blueprintId";

    private final BlueprintStateController contentBlueprintStateController;
    private final SpaceBlueprintStateController spaceBlueprintStateController;

    private final TransactionTemplate transactionTemplate;

    public BlueprintStateResource(
            final BlueprintStateController contentBlueprintStateController,
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport TransactionTemplate transactionTemplate,
            final @ComponentImport PermissionManager permissionManager,
            final SpaceBlueprintStateController spaceBlueprintStateController,
            final @ComponentImport AccessModeService accessModeService) {
        super(permissionManager, spaceManager, accessModeService);

        this.contentBlueprintStateController = contentBlueprintStateController;
        this.transactionTemplate = transactionTemplate;
        this.spaceBlueprintStateController = spaceBlueprintStateController;
    }

    @Path("/{" + PARAM_BLUEPRINT_ID + "}")
    @PUT
    @ReadOnlyAccessAllowed
    public void enableBlueprint(@PathParam(PARAM_BLUEPRINT_ID) final String blueprintId,
                                @QueryParam(PARAM_SPACE_KEY) String spaceKey) {
        toggleContentBlueprint(blueprintId, spaceKey, true);
    }

    @Path("/{" + PARAM_BLUEPRINT_ID + "}")
    @DELETE
    @ReadOnlyAccessAllowed
    public void disableBlueprint(@PathParam(PARAM_BLUEPRINT_ID) String blueprintId,
                                 @QueryParam(PARAM_SPACE_KEY) String spaceKey) {
        toggleContentBlueprint(blueprintId, spaceKey, false);
    }

    @Path("/space-blueprint/{" + PARAM_BLUEPRINT_ID + "}")
    @PUT
    @ReadOnlyAccessAllowed
    public void enableSpaceBlueprint(@PathParam(PARAM_BLUEPRINT_ID) final String blueprintId) {
        toggleSpaceBlueprint(blueprintId, true);
    }

    @Path("/space-blueprint/{" + PARAM_BLUEPRINT_ID + "}")
    @DELETE
    @ReadOnlyAccessAllowed
    public void disableSpaceBlueprint(@PathParam(PARAM_BLUEPRINT_ID) String blueprintId) {
        toggleSpaceBlueprint(blueprintId, false);
    }

    private UUID validateAndGetId(String blueprintId) {
        checkEmptyParameter(blueprintId, PARAM_BLUEPRINT_ID);

        return UUID.fromString(blueprintId);
    }

    private Space validatePermissionsAndGetSpace(String spaceKey) {
        final Space space;

        if (StringUtils.isBlank(spaceKey)) {
            checkAdminPermission();
            space = null;
        } else {
            space = checkSpaceAdminPermission(spaceKey);
        }

        return space;
    }

    private void toggleContentBlueprint(final String blueprintId, String spaceKey, final boolean enable) {
        final UUID id = validateAndGetId(blueprintId);
        final Space space = validatePermissionsAndGetSpace(spaceKey);

        transactionTemplate.execute(() -> {
            if (enable)
                contentBlueprintStateController.enableBlueprint(id, space);
            else
                contentBlueprintStateController.disableBlueprint(id, space);

            return null;
        });
    }

    private void toggleSpaceBlueprint(final String blueprintId, final boolean enable) {
        final UUID id = validateAndGetId(blueprintId);
        validatePermissionsAndGetSpace(null);

        transactionTemplate.execute(() -> {
            if (enable)
                spaceBlueprintStateController.enableSpaceBlueprint(id);
            else
                spaceBlueprintStateController.disableSpaceBlueprint(id);

            return null;
        });
    }

}
