package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.confluence.plugins.createcontent.PluginSpaceBlueprintAccessor;
import com.atlassian.confluence.plugins.createcontent.SpaceBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.activeobjects.ContentTemplateRefAo;
import com.atlassian.confluence.plugins.createcontent.activeobjects.SpaceBlueprintAo;
import com.atlassian.confluence.plugins.dialog.wizard.api.DialogWizard;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.google.common.base.Joiner;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import static com.google.common.collect.Collections2.transform;
import static com.google.common.collect.ImmutableList.copyOf;
import static com.google.common.collect.Lists.newArrayList;
import static java.util.Arrays.asList;

/**
 * Finds SpaceBlueprints in the AO table. If not found, gets from the plugin system.
 *
 * @since 1.6
 */
@Component("spaceBlueprintManager")
public class DefaultSpaceBlueprintManager extends AbstractAoManager<SpaceBlueprint, SpaceBlueprintAo>
        implements SpaceBlueprintManager {
    private final ConcurrentHashMap<ModuleCompleteKey, ModuleCompleteKey> getOrCreateLocks = new ConcurrentHashMap<>();

    private final PluginSpaceBlueprintAccessor delegate;
    private final DefaultContentTemplateRefManager contentTemplateRefManager;

    @Autowired
    public DefaultSpaceBlueprintManager(
            final PluginSpaceBlueprintAccessor delegate,
            final @ComponentImport ActiveObjects activeObjects,
            final DefaultContentTemplateRefManager contentTemplateRefManager) {
        super(activeObjects, SpaceBlueprintAo.class);

        this.delegate = delegate;
        this.contentTemplateRefManager = contentTemplateRefManager;
    }

    @Nonnull
    @Override
    public SpaceBlueprintAo[] getAosByModuleCompleteKey(@Nonnull final ModuleCompleteKey moduleCompleteKey,
                                                        final boolean pluginClone) {
        // CONFDEV-19787: If we don't synchronize, we can end up having multiple clones in the DB
        Object lock = getOrCreateLocks.putIfAbsent(moduleCompleteKey, moduleCompleteKey);
        if (lock == null) {
            lock = moduleCompleteKey;
        }
        synchronized (lock) {
            return activeObjects.executeInTransaction(() -> {
                SpaceBlueprintAo[] aoBlueprints = internalGetAosByModuleCompleteKey(moduleCompleteKey, pluginClone);
                if (aoBlueprints.length == 0) {
                    SpaceBlueprint pluginBlueprint = delegate.getByModuleCompleteKey(moduleCompleteKey);
                    aoBlueprints = new SpaceBlueprintAo[]{internalCreateAo(pluginBlueprint)};
                }
                return aoBlueprints;
            });
        }
    }

    @Nonnull
    @Override
    public List<SpaceBlueprint> getAll() {
        List<SpaceBlueprint> aoBlueprints = getAoSpaceBlueprints();

        List<SpaceBlueprint> pluginBlueprints = delegate.getAll();
        if (addPluginBlueprintsToAo(aoBlueprints, pluginBlueprints)) {
            // Refresh from the AO store.
            aoBlueprints = getAoSpaceBlueprints();
        }

        return aoBlueprints;
    }

    @Override
    protected void internalDeleteAo(@Nonnull final SpaceBlueprintAo ao) {
        ContentTemplateRefAo homePage = ao.getHomePage();

        ao.setHomePage(null);
        ao.setCategory(null);
        ao.save();

        if (homePage != null) {
            // FIXME: Calling delete is what we should be doing, but that will initiate another transaction right now
            contentTemplateRefManager.internalDeleteAo(homePage);
            activeObjects.delete(homePage);
        }
    }

    private boolean addPluginBlueprintsToAo(List<SpaceBlueprint> aoBlueprints, List<SpaceBlueprint> pluginBlueprints) {
        boolean pluginBlueprintsAdded = false;

        for (SpaceBlueprint pluginBlueprint : pluginBlueprints) {
            if (findPluginBlueprint(pluginBlueprint.getModuleCompleteKey(), aoBlueprints) == null) {
                createAo(pluginBlueprint);
                pluginBlueprintsAdded = true;
            }
        }

        return pluginBlueprintsAdded;
    }

    private SpaceBlueprint findPluginBlueprint(String moduleKey, List<SpaceBlueprint> aoBlueprints) {
        for (SpaceBlueprint aoBlueprint : aoBlueprints) {
            if (aoBlueprint.isPluginClone() && moduleKey.equals(aoBlueprint.getModuleCompleteKey())) {
                // Found it, nothing to do.
                return aoBlueprint;
            }
        }
        return null;
    }

    private List<SpaceBlueprint> getAoSpaceBlueprints() {
        return activeObjects.executeInTransaction((TransactionCallback<List<SpaceBlueprint>>) () -> {
            List<SpaceBlueprintAo> aos = asList(activeObjects.find(SpaceBlueprintAo.class));
            return copyOf(transform(aos, this::build));
        });
    }

    @Nonnull
    @Override
    protected SpaceBlueprintAo internalCreateAo(@Nonnull SpaceBlueprint original) {
        // Create space blueprint
        SpaceBlueprintAo ao = helperAoManager.createWithUuid();

        final UUID homePageId = original.getHomePageId();
        if (homePageId != null) {
            // A bit lame that we're getting this object from the DB just to save the reference ID, although this also makes sure the object really exists
            // TODO: Do we check if the object could be correctly retrieved?
            ao.setHomePage(contentTemplateRefManager.internalGetAoById(homePageId));
        }

        copyPropertiesIntoAo(ao, original, true);
        ao.save();

        return ao;
    }

    @Nonnull
    @Override
    protected SpaceBlueprintAo internalUpdateAo(@Nonnull final SpaceBlueprint object) {
        SpaceBlueprintAo ao = internalGetAoById(object.getId());
        if (ao == null) {
            String error = String.format("Space Blueprint with UUID %s not found", object.getId());
            throw new IllegalStateException(error);
        }

        final UUID homePageId = object.getHomePageId();

        ao.setHomePage(homePageId == null ? null : contentTemplateRefManager.getAoById(homePageId));

        copyPropertiesIntoAo(ao, object, false);
        ao.save();

        return ao;
    }

    private void copyPropertiesIntoAo(@Nonnull SpaceBlueprintAo ao, @Nonnull SpaceBlueprint original, boolean isCreate) {
        if (isCreate) {
            ao.setPluginModuleKey(original.getModuleCompleteKey());
        }
        ao.setI18nNameKey(original.getI18nNameKey());
        ao.setPluginClone(original.isPluginClone());
        ao.setCategory(original.getCategory());

        final List<ModuleCompleteKey> promotedBps = original.getPromotedBps();
        if (promotedBps != null && !promotedBps.isEmpty()) {
            ao.setPromotedBps(Joiner.on(',').join(promotedBps));
        } else {
            ao.setPromotedBps(null);
        }
    }

    @Nonnull
    @Override
    protected SpaceBlueprint build(@Nonnull SpaceBlueprintAo ao) {
        final ContentTemplateRefAo homePage = ao.getHomePage();
        final UUID homePageId = homePage != null ? UUID.fromString(homePage.getUuid()) : null;

        String pluginModuleKey = ao.getPluginModuleKey();
        String category = ao.getCategory();

        List<ModuleCompleteKey> promotedBps = newArrayList();
        String promotedBpsStr = ao.getPromotedBps();
        if (promotedBpsStr != null && !promotedBpsStr.isEmpty()) {
            String[] arr = promotedBpsStr.split(",");
            for (String promotedBp : arr) {
                promotedBps.add(new ModuleCompleteKey(promotedBp));
            }
        }

        DialogWizard dialogWizard = delegate.getDialogByModuleCompleteKey(new ModuleCompleteKey(pluginModuleKey));
        SpaceBlueprint result = new SpaceBlueprint(UUID.fromString(ao.getUuid()), pluginModuleKey, ao.getI18nNameKey(),
                ao.isPluginClone(), promotedBps, dialogWizard, category);
        result.setHomePageId(homePageId);
        return result;
    }

    @Override
    public SpaceBlueprint create(@Nonnull final SpaceBlueprint original, @Nullable final UUID homePageId) {
        if (homePageId != null) {
            original.setHomePageId(homePageId);
        }
        return create(original);
    }
}
