package com.atlassian.confluence.plugins.createcontent;

import com.atlassian.confluence.security.SpacePermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.spaces.SpacesQuery;
import com.atlassian.confluence.user.ConfluenceUser;
import com.google.common.base.Predicate;

import java.util.Collection;
import java.util.Set;

import static com.atlassian.confluence.security.SpacePermission.CREATEEDIT_PAGE_PERMISSION;
import static com.atlassian.confluence.security.SpacePermission.EDITBLOG_PERMISSION;
import static com.atlassian.confluence.spaces.SpaceStatus.CURRENT;
import static com.google.common.collect.Sets.newLinkedHashSet;
import static com.google.common.collect.Sets.union;

public class SpaceUtils {
    public static boolean hasEditableSpaces(ConfluenceUser user, SpaceManager spaceManager) {
        return !getSpacesForPermission(user, 1, CREATEEDIT_PAGE_PERMISSION, spaceManager).isEmpty()
                || !getSpacesForPermission(user, 1, EDITBLOG_PERMISSION, spaceManager).isEmpty();
    }

    public static boolean hasCreatePagePermission(ConfluenceUser user, Space space, SpaceManager spaceManager, SpacePermissionManager spacePermissionManager) {
        if (space != null) {
            return spacePermissionManager.hasPermission(CREATEEDIT_PAGE_PERMISSION, space, user);
        } else {
            return !getSpacesForPermission(user, 1, CREATEEDIT_PAGE_PERMISSION, spaceManager).isEmpty();
        }
    }

    public static Predicate<Space> editableSpaceFilter(final ConfluenceUser user, final SpacePermissionManager spacePermissionManager, final String spacePermission) {
        return space -> {
            if (spacePermission != null) {
                return space != null &&
                        spacePermissionManager.hasPermission(spacePermission, space, user);
            } else {
                return space != null &&
                        spacePermissionManager.hasPermission(CREATEEDIT_PAGE_PERMISSION, space, user) ||
                        spacePermissionManager.hasPermission(EDITBLOG_PERMISSION, space, user);
            }
        };
    }

    public static Collection<Space> getEditableSpaces(ConfluenceUser user, int limit, SpaceManager spaceManager, String spacePermission) {
        if (spacePermission != null) {
            return getSpacesForPermission(user, limit, spacePermission, spaceManager);
        } else {
            // This would be better done as a single query with both permissions, but SpacesQuery doesn't support that
            return union(
                    getSpacesForPermission(user, limit, CREATEEDIT_PAGE_PERMISSION, spaceManager),
                    getSpacesForPermission(user, limit, EDITBLOG_PERMISSION, spaceManager)
            );
        }
    }

    private static Set<Space> getSpacesForPermission(ConfluenceUser user, int limit, String permission, SpaceManager spaceManager) {
        final SpacesQuery spacesQuery = SpacesQuery.newQuery()
                .forUser(user)
                .withSpaceStatus(CURRENT)
                .withPermission(permission)
                .unsorted() // Just need to find out if any exists, order does not matter here
                .build();

        return newLinkedHashSet(spaceManager.getSpaces(spacesQuery).getPage(0, limit));
    }
}