package com.atlassian.confluence.plugins.createcontent.actions;

import com.atlassian.confluence.core.DefaultSaveContext;
import com.atlassian.confluence.labels.Label;
import com.atlassian.confluence.labels.LabelManager;
import com.atlassian.confluence.labels.Labelable;
import com.atlassian.confluence.labels.Namespace;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.pages.PageUpdateTrigger;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PluginTemplateReference;
import com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContext;
import com.atlassian.confluence.plugins.createcontent.extensions.BlueprintDescriptor;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.plugins.createcontent.template.PluginPageTemplateHelper;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.spring.scanner.annotation.export.ExportAsService;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.message.I18nResolver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;

import static com.atlassian.confluence.labels.Namespace.getNamespace;
import static com.atlassian.confluence.pages.templates.PluginTemplateReference.spaceTemplateReference;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.INDEX_KEY;
import static java.lang.String.format;

/**
 * I look after creation and lookup of Blueprint index pages
 */
@Component
@ExportAsService(IndexPageManager.class)
public class DefaultIndexPageManager implements IndexPageManager {
    private static final String INDEX_PAGE_LABEL = "blueprint-index-page";

    private final PageManager pageManager;
    private final BlueprintContentGenerator contentGenerator;
    private final LabelManager labelManager;
    private final I18nResolver i18nResolver;
    private final PluginPageTemplateHelper pluginPageTemplateHelper;

    @Autowired
    public DefaultIndexPageManager(
            final @ComponentImport PageManager pageManager,
            final BlueprintContentGenerator contentGenerator,
            final @ComponentImport LabelManager labelManager,
            final @ComponentImport I18nResolver i18nResolver,
            final PluginPageTemplateHelper pluginPageTemplateHelper) {
        this.pageManager = pageManager;
        this.contentGenerator = contentGenerator;
        this.labelManager = labelManager;
        this.i18nResolver = i18nResolver;
        this.pluginPageTemplateHelper = pluginPageTemplateHelper;
    }

    /**
     * @deprecated since 2.0.0
     */
    @Override
    @Deprecated
    public Page getOrCreateIndexPage(final BlueprintDescriptor blueprintDescriptor, final Space space, final String desiredTitle) {
        final Page existingIndexPage = findIndexPage(blueprintDescriptor.getBlueprintKey(), space);
        if (existingIndexPage != null) {
            return existingIndexPage;
        } else {
            return createIndexPage(blueprintDescriptor, space, desiredTitle);
        }
    }

    @Override
    public Page getOrCreateIndexPage(final ContentBlueprint blueprint, final Space space, final String desiredTitle) {
        final Page existingIndexPage = findIndexPage(new ModuleCompleteKey(blueprint.getModuleCompleteKey()), space);
        if (existingIndexPage != null) {
            return existingIndexPage;
        } else {
            return createIndexPage(blueprint, space, desiredTitle);
        }
    }

    /**
     * @deprecated since 2.0.0
     */
    @Deprecated
    private Page createIndexPage(BlueprintDescriptor blueprintDescriptor, Space space, String desiredTitle) {
        // For this deprecated code to work, we have to use the "old" method of resolving a page template from a
        // content-template module key, a space, and a content blueprint module key
        final ModuleCompleteKey indexTemplate = blueprintDescriptor.getIndexTemplate();
        PluginTemplateReference pluginTemplateReference = spaceTemplateReference(indexTemplate, blueprintDescriptor.getBlueprintKey(), space);
        PageTemplate pageTemplate = pluginPageTemplateHelper.getPageTemplate(pluginTemplateReference);
        ContentTemplateRef indexTemplateRef = new ContentTemplateRef(null, pageTemplate.getId(), indexTemplate.getCompleteKey(), pageTemplate.getName(), false, null);

        ContentBlueprint bp = new ContentBlueprint();
        bp.setIndexPageTemplateRef(indexTemplateRef);
        bp.setIndexKey(blueprintDescriptor.getIndexKey());
        bp.setModuleCompleteKey(blueprintDescriptor.getBlueprintKey().getCompleteKey());
        bp.setCreateResult(blueprintDescriptor.getCreateResult());

        return createIndexPage(bp, space, desiredTitle);
    }

    @Override
    public Page createIndexPage(ContentBlueprint blueprint, Space space, String desiredTitle) {
        ModuleCompleteKey blueprintKey = new ModuleCompleteKey(blueprint.getModuleCompleteKey());
        final int maxAttempts = 3;
        int attempt = 0;
        while (++attempt <= maxAttempts) {
            final Page newIndexPage = createIndexPageObject(blueprint, space, desiredTitle);
            final Label label = getIndexPageSystemLabel(blueprintKey);
            labelManager.addLabel((Labelable) newIndexPage, label);

            String title = getIndexPageTitle(attempt, desiredTitle);

            if (pageManager.getPage(space.getKey(), title) == null) {
                newIndexPage.setTitle(title);
                pageManager.saveContentEntity(newIndexPage, new DefaultSaveContext(true, true, false, PageUpdateTrigger.SPACE_CREATE));
                return newIndexPage;
            }
        }
        throw new IllegalStateException("Failed to create index page for " + blueprintKey + " after " + maxAttempts + " attempts");
    }

    private static String getIndexPageTitle(int attempt, String desiredTitle) {
        if (attempt == 1) {
            return desiredTitle;
        } else {
            return format("%s (%s)", desiredTitle, attempt);
        }
    }

    @Override
    public Page findIndexPage(ContentBlueprint blueprint, Space space) {
        return findIndexPage(new ModuleCompleteKey(blueprint.getModuleCompleteKey()), space);
    }

    private Page findIndexPage(ModuleCompleteKey moduleCompleteKey, Space space) {
        final Label label = getIndexPageSystemLabel(moduleCompleteKey);
        final List<? extends Labelable> content = labelManager.getCurrentContentForLabelAndSpace(label, space.getKey());

        // What happens if we get multiple hits here?  Should never happen, but.....
        for (Labelable labelable : content) {
            if (labelable instanceof Page) {
                return (Page) labelable;
            }
        }
        return null;
    }

    private static Label getIndexPageSystemLabel(ModuleCompleteKey moduleCompleteKey) {
        return new Label(INDEX_PAGE_LABEL, getBlueprintLabelNamespace(moduleCompleteKey));
    }

    private static Namespace getBlueprintLabelNamespace(ModuleCompleteKey moduleCompleteKey) {
        return getNamespace(moduleCompleteKey.getCompleteKey());
    }

    private Page createIndexPageObject(ContentBlueprint blueprint, Space space, String desiredTitle) {
        String indexKey = blueprint.getIndexKey();
        ContentTemplateRef contentTemplateRef = blueprint.getIndexPageTemplateRef();
        ModuleCompleteKey blueprintKey = new ModuleCompleteKey(blueprint.getModuleCompleteKey());

        String createFromTemplateLabel = i18nResolver.getText(
                "com.atlassian.confluence.plugins.confluence-create-content-plugin.create-from-template.default-index.label",
                desiredTitle
        );

        BlueprintContext context = new BlueprintContext();
        context.setTemplateLabel(indexKey);
        context.setAnalyticsKey(indexKey);     // HACK - same thing as label!
        context.setSpaceKey(space.getKey());
        context.setBlueprintId(blueprint.getId());
        context.setBlueprintModuleCompleteKey(blueprintKey);
        context.setCreateFromTemplateLabel(createFromTemplateLabel);
        context.put(INDEX_KEY, contentTemplateRef.getModuleCompleteKey());

        return contentGenerator.createIndexPageObject(contentTemplateRef, space, context.getMap());
    }
}
