package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.confluence.spaces.Space;
import com.google.common.base.Predicate;

import java.util.Collection;
import java.util.Set;

import static com.atlassian.confluence.plugins.createcontent.rest.entities.SpaceEntity.spaceTransformer;
import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Collections2.transform;
import static com.google.common.collect.Sets.newLinkedHashSet;
import static java.util.Arrays.asList;

public class SpaceResultsEntityBuilder {
    private final Set<Space> spaces = newLinkedHashSet();
    private final int resultsLimit;
    private final Predicate<? super Space> spaceFilter;

    private boolean allSpacesAdded = true;

    public SpaceResultsEntityBuilder(int resultsLimit, Predicate<Space> spaceFilter) {
        this.resultsLimit = resultsLimit;
        this.spaceFilter = spaceFilter;
    }

    /**
     * Adds the contents of one collection to another, one item at a time, until the size of the destination collection
     * reaches a certain limit.
     *
     * @return true if all items could be added to the destination, false otherwise
     */
    private static <T> boolean addUntilLimitReached(Collection<T> destination, Iterable<T> source, int destinationSizeLimit) {
        for (T item : source) {
            if (destination.size() < destinationSizeLimit) {
                destination.add(item);
            } else {
                return false;
            }
        }
        return true;
    }

    public SpaceResultsEntityBuilder addSpaces(Space... spaces) {
        return addSpaces(asList(spaces));
    }

    public SpaceResultsEntityBuilder addSpaces(Collection<Space> newSpaces) {
        allSpacesAdded &= addUntilLimitReached(this.spaces, filter(newSpaces, spaceFilter), resultsLimit);
        return this;
    }

    public Set<Space> getSpaces() {
        return spaces;
    }

    public SpaceResultsEntity build() {
        return new SpaceResultsEntity(transform(spaces, spaceTransformer()), resultsLimit, !allSpacesAdded);
    }
}
