package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.bonnie.Searchable;
import com.atlassian.confluence.labels.LabelManager;
import com.atlassian.confluence.macro.query.BooleanQueryFactory;
import com.atlassian.confluence.plugins.createcontent.rest.SpaceResultsEntity;
import com.atlassian.confluence.plugins.createcontent.rest.SpaceResultsEntityBuilder;
import com.atlassian.confluence.plugins.createcontent.services.SpaceCollectionService;
import com.atlassian.confluence.search.v2.ContentSearch;
import com.atlassian.confluence.search.v2.SearchManager;
import com.atlassian.confluence.search.v2.filter.SubsetResultFilter;
import com.atlassian.confluence.search.v2.query.BooleanQuery;
import com.atlassian.confluence.search.v2.query.ContentTypeQuery;
import com.atlassian.confluence.search.v2.query.CreatorQuery;
import com.atlassian.confluence.search.v2.searchfilter.SiteSearchPermissionsSearchFilter;
import com.atlassian.confluence.security.SpacePermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.spaces.Spaced;
import com.atlassian.confluence.spaces.SpacesQuery;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.atlassian.confluence.plugins.createcontent.SpaceUtils.editableSpaceFilter;
import static com.atlassian.confluence.plugins.createcontent.SpaceUtils.getEditableSpaces;
import static com.atlassian.confluence.search.service.ContentTypeEnum.BLOG;
import static com.atlassian.confluence.search.service.ContentTypeEnum.PAGE;
import static com.atlassian.confluence.search.v2.SearchManager.EntityVersionPolicy.LATEST_VERSION;
import static com.atlassian.confluence.search.v2.sort.CreatedSort.DESCENDING;
import static com.google.common.base.Predicates.in;
import static com.google.common.base.Predicates.not;
import static com.google.common.collect.Sets.newLinkedHashSet;
import static java.util.Collections.emptyList;
import static java.util.EnumSet.of;
import static org.slf4j.LoggerFactory.getLogger;

@Component
public class DefaultSpaceCollectionService implements SpaceCollectionService {

    private static final Logger log = getLogger(DefaultSpaceCollectionService.class);
    private final SpacePermissionManager spacePermissionManager;
    private final SpaceManager spaceManager;
    private final LabelManager labelManager;
    private final SearchManager searchManager;

    @Autowired
    public DefaultSpaceCollectionService(
            final @ComponentImport SpacePermissionManager spacePermissionManager,
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport LabelManager labelManager,
            final @ComponentImport SearchManager searchManager) {
        this.spacePermissionManager = spacePermissionManager;
        this.spaceManager = spaceManager;
        this.labelManager = labelManager;
        this.searchManager = searchManager;
    }

    public Map<String, SpaceResultsEntity> getSpaces(List<String> promotedSpaceKeys, int promotedSpacesLimit, int otherSpacesLimit, String spacePermission) {
        final ConfluenceUser user = AuthenticatedUserThreadLocal.get();

        SpaceResultsEntityBuilder promotedSpaces = getPromotedSpaces(user, promotedSpaceKeys, promotedSpacesLimit, spacePermission);
        SpaceResultsEntityBuilder otherSpaces = getOtherSpaces(user, otherSpacesLimit, promotedSpaces.getSpaces(), spacePermission);

        return ImmutableMap
                .of("promotedSpaces", promotedSpaces.build(), "otherSpaces", otherSpaces.build());
    }

    /**
     * @return a potent blend of favourite, recently edited and required spaces
     */
    private SpaceResultsEntityBuilder getPromotedSpaces(ConfluenceUser user, List<String> requiredSpaceKeys, int promotedSpacesLimit, String spacePermission) {
        final Predicate<Space> spacePermissionsFilter = editableSpaceFilter(user, spacePermissionManager, spacePermission);
        final SpaceResultsEntityBuilder spaceResultsBuilder = new SpaceResultsEntityBuilder(promotedSpacesLimit, spacePermissionsFilter);

        if (requiredSpaceKeys != null && !requiredSpaceKeys.isEmpty()) {
            spaceResultsBuilder.addSpaces(getRequiredSpaces(requiredSpaceKeys, promotedSpacesLimit));
        }

        if (user != null) {
            // Casting to User because we want to be Confluence 5.2 compatible, if possible
            final Space personalSpace = spaceManager.getPersonalSpace(user);
            if (personalSpace != null) {
                spaceResultsBuilder.addSpaces(personalSpace);
            }

            spaceResultsBuilder
                    .addSpaces(getRecentContentSpaces(user))
                    .addSpaces(labelManager.getFavouriteSpaces(user.getName()));
        }

        return spaceResultsBuilder;
    }

    private SpaceResultsEntityBuilder getOtherSpaces(ConfluenceUser user, int resultsLimit, Collection<Space> excludedSpaces, String spacePermission) {
        // need to fetch enough editable spaces so that even after having the excluded spaces filtered out, the
        // collection will still reach the resultsLimit in size.  Add +1 to this so that we fetch one more than we can
        // use, so we know if we're truncating the results or not.
        int editableSpacesQueryLimit = resultsLimit + excludedSpaces.size() + 1;
        final List<Space> editableSpaces = Lists.newArrayList(getEditableSpaces(user, editableSpacesQueryLimit, spaceManager, spacePermission));

        final Predicate<Space> spacesFilter = not(in(excludedSpaces));

        editableSpaces.sort((o1, o2) -> o1.getName().compareToIgnoreCase(o2.getName()));
        return new SpaceResultsEntityBuilder(resultsLimit, spacesFilter).addSpaces(editableSpaces);
    }

    private List<Space> getRequiredSpaces(List<String> requiredSpaceKeys, int promotedSpacesLimit) {
        final SpacesQuery query = SpacesQuery.newQuery().withSpaceKeys(requiredSpaceKeys).build();
        final List<Space> spaces = spaceManager.getSpaces(query).getPage(0, promotedSpacesLimit);

        // explicitly sort the required spaces to be in the same order that they were asked for
        spaces.sort(sortByKeyOrder(requiredSpaceKeys));
        return spaces;
    }

    private static Comparator<Space> sortByKeyOrder(final List<String> keyOrder) {
        return (space1, space2) -> keyOrder.indexOf(space1.getKey()) - keyOrder.indexOf(space2.getKey());
    }

    private Collection<Space> getRecentContentSpaces(final ConfluenceUser user) {
        final Set<Space> spaces = newLinkedHashSet();
        for (Searchable searchable : searchForRecentContent(user)) {
            if (searchable instanceof Spaced) {
                Space space = ((Spaced) searchable).getSpace();

                if (space != null) {
                    spaces.add(space);
                }

            }
        }
        return spaces;
    }

    private Iterable<Searchable> searchForRecentContent(ConfluenceUser user) {
        try {
            final ContentSearch search = new ContentSearch(
                    recentContentQuery(user),
                    DESCENDING,
                    SiteSearchPermissionsSearchFilter.getInstance(),
                    new SubsetResultFilter(25)
            );
            return searchManager.searchEntities(search, LATEST_VERSION);
        } catch (Exception e) {
            log.error("Error when searching for recent content", e);
            return emptyList();
        }
    }

    private BooleanQuery recentContentQuery(ConfluenceUser user) {
        BooleanQueryFactory booleanQueryFactory = new BooleanQueryFactory();
        booleanQueryFactory.addMust(new CreatorQuery(user.getKey()));
        booleanQueryFactory.addMust(new ContentTypeQuery(of(PAGE, BLOG)));

        return booleanQueryFactory.toBooleanQuery();
    }
}
