package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.ContentStatus;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.content.template.ContentBlueprintInstance;
import com.atlassian.confluence.api.model.link.LinkType;
import com.atlassian.confluence.api.service.content.ContentDraftService;
import com.atlassian.confluence.api.service.exceptions.BadRequestException;
import com.atlassian.confluence.core.ContentEntityManager;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.core.ContentPermissionManager;
import com.atlassian.confluence.core.DefaultSaveContext;
import com.atlassian.confluence.labels.Label;
import com.atlassian.confluence.labels.LabelManager;
import com.atlassian.confluence.labels.Labelable;
import com.atlassian.confluence.pages.Draft;
import com.atlassian.confluence.pages.DraftManager;
import com.atlassian.confluence.pages.DraftsTransitionHelper;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.pages.actions.PagePermissionsActionHelper;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PageTemplateManager;
import com.atlassian.confluence.plugins.createcontent.BlueprintConstants;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.ContentTemplateRefManager;
import com.atlassian.confluence.plugins.createcontent.actions.BlueprintContentGenerator;
import com.atlassian.confluence.plugins.createcontent.actions.BlueprintManager;
import com.atlassian.confluence.plugins.createcontent.api.events.BlueprintPageCreateEvent;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.BlueprintIllegalArgumentException;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.api.services.ContentBlueprintService;
import com.atlassian.confluence.plugins.createcontent.extensions.UserBlueprintConfigManager;
import com.atlassian.confluence.plugins.createcontent.services.RequestResolver;
import com.atlassian.confluence.plugins.createcontent.services.RequestStorage;
import com.atlassian.confluence.plugins.createcontent.services.model.BlueprintPage;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageEntity;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageRequest;
import com.atlassian.confluence.security.ContentPermission;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.confluence.xwork.FlashScope;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.spring.scanner.annotation.export.ExportAsService;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import javax.annotation.Nonnull;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static com.atlassian.confluence.api.model.content.ContentStatus.CURRENT;
import static com.atlassian.confluence.api.model.content.ContentStatus.DRAFT;
import static com.atlassian.confluence.api.service.content.ContentDraftService.ConflictPolicy.ABORT;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.CREATE_RESULT_VIEW;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @since 1.6
 */
@ExportAsService({ContentBlueprintService.class, com.atlassian.confluence.api.service.content.ContentBlueprintService.class})
@Component
public class DefaultContentBlueprintService implements ContentBlueprintService, com.atlassian.confluence.api.service.content.ContentBlueprintService {
    private final ContentBlueprintManager contentBlueprintManager;
    private final BlueprintManager legacyManager;
    private final PageManager pageManager;
    private final DraftManager draftManager;
    private final BlueprintContentGenerator contentGenerator;
    private final LabelManager labelManager;
    private final RequestResolver requestResolver;
    private final EventPublisher eventPublisher;
    private final ContentPermissionManager contentPermissionManager;
    private final RequestStorage requestStorage;
    private final DraftsTransitionHelper draftsTransitionHelper;
    private final UserBlueprintConfigManager userBlueprintConfigManager;
    private final ContentDraftService contentDraftService;
    private final ContentEntityManager contentEntityManager;
    private final SpaceManager spaceManager;
    private final ContentBlueprintInstanceAdapter contentBlueprintInstanceAdapter;
    private final UserAccessor userAccessor;
    private final ContentTemplateRefManager contentTemplateRefManager;
    private final PageTemplateManager pageTemplateManager;

    @VisibleForTesting
    static final String VIEW_PERMISSIONS_USERS = "viewPermissionsUsers";

    @Autowired
    public DefaultContentBlueprintService(
            final ContentBlueprintManager contentBlueprintManager,
            final BlueprintManager blueprintManager,
            final @ComponentImport PageManager pageManager,
            final @ComponentImport DraftManager draftManager,
            final BlueprintContentGenerator contentGenerator,
            final @ComponentImport LabelManager labelManager,
            final RequestResolver requestResolver,
            final @ComponentImport EventPublisher eventPublisher,
            final @ComponentImport ContentPermissionManager contentPermissionManager,
            final RequestStorage requestStorage,
            final @ComponentImport DraftsTransitionHelper draftsTransitionHelper,
            final UserBlueprintConfigManager userBlueprintConfigManager,
            final @ComponentImport ContentDraftService contentDraftService,
            final @ComponentImport @Qualifier("contentEntityManager") ContentEntityManager contentEntityManager,
            final @ComponentImport SpaceManager spaceManager,
            final ContentBlueprintInstanceAdapter contentBlueprintInstanceAdapter,
            final @ComponentImport UserAccessor userAccessor,
            final ContentTemplateRefManager contentTemplateRefManager,
            final @ComponentImport PageTemplateManager pageTemplateManager) {
        this.contentBlueprintManager = contentBlueprintManager;
        this.legacyManager = blueprintManager;
        this.pageManager = pageManager;
        this.draftManager = draftManager;
        this.contentGenerator = contentGenerator;
        this.labelManager = labelManager;
        this.requestResolver = requestResolver;
        this.eventPublisher = eventPublisher;
        this.contentPermissionManager = contentPermissionManager;
        this.requestStorage = requestStorage;
        this.draftsTransitionHelper = draftsTransitionHelper;
        this.userBlueprintConfigManager = userBlueprintConfigManager;
        this.contentDraftService = contentDraftService;
        this.contentEntityManager = contentEntityManager;
        this.spaceManager = spaceManager;
        this.contentBlueprintInstanceAdapter = contentBlueprintInstanceAdapter;
        this.userAccessor = userAccessor;
        this.contentTemplateRefManager = contentTemplateRefManager;
        this.pageTemplateManager = pageTemplateManager;
    }

    @Override
    public ContentBlueprintInstance createInstance(ContentBlueprintInstance contentBlueprintInstance, Expansion... expansions) {
        ContentStatus status = contentBlueprintInstance.getContent().getStatus();
        if (!ImmutableList.of(DRAFT, CURRENT).contains(status))
            throw new BadRequestException("Status of content must be DRAFT or CURRENT, supplied value: " + status);

        CreateBlueprintPageEntity entity = contentBlueprintInstanceAdapter.convertToEntity(contentBlueprintInstance);
        try {
            final ConfluenceUser creator = AuthenticatedUserThreadLocal.get();

            if (status.equals(DRAFT)) {
                ContentEntityObject created = createContentDraft(entity, creator);
                return contentBlueprintInstanceAdapter.convertToInstance(created, contentBlueprintInstance, expansions);
            } else {
                BlueprintPage created = createPage(entity, creator);
                return contentBlueprintInstanceAdapter.convertToInstance(created, contentBlueprintInstance, expansions);
            }
        } catch (BlueprintIllegalArgumentException e) {
            throw new BadRequestException("It's bad, real bad", e);
        }
    }

    @Override
    public Content publishInstance(Content content, Expansion... expansions) {
        if (content.getId() == null) {
            throw new BadRequestException("Require draft id");
        }

        if (!content.getStatus().equals(CURRENT)) {
            throw new BadRequestException("Status is not supported: " + content.getStatus().toString());
        }

        ContentEntityObject draft = getDraft(content.getId());
        if (draft == null) {
            throw new BadRequestException("Could not find draft to publish with id: " + content.getId().asLong());
        }

        CreateBlueprintPageEntity createBlueprintPageEntity = getCreateBlueprintPageEntity(draft);
        ContentBlueprint blueprint = getBlueprint(createBlueprintPageEntity);

        if (blueprint != null) {
            content = createIndexPageAndSetAsParent(content, blueprint);
        }

        // Clear the createBlueprintPageEntity from the DB after it is retrieved to reconstruct the blueprint object
        requestStorage.clear(draft);

        Content createdContent;
        if (draftsTransitionHelper.isSharedDraftsFeatureEnabled(content.getSpace().getKey())) {
            createdContent = contentDraftService.publishEditDraft(content, ABORT);
        } else {
            createdContent = contentDraftService.publishNewDraft(content, expansions);
        }

        if (blueprint != null) {
            ConfluenceUser user = AuthenticatedUserThreadLocal.get();
            Page createdPage = pageManager.getPage(createdContent.getId().asLong());

            //Make a check to see if need to show the blueprint index popup and decorate return url with flash-id
            if (user != null && userBlueprintConfigManager.isFirstBlueprintOfTypeForUser(blueprint.getId(), user)) {
                String webUILink = createdContent.getLinks().get(LinkType.WEB_UI).getPath();
                String url = decorateReturnedUrlWithFlashScopeInfo(blueprint, user, webUILink);
                createdContent = Content.builder(createdContent).addLink(LinkType.WEB_UI, url).build();
            }

            //Fire event publish blueprint page
            Map<String, Object> emptyContext = createBlueprintPageEntity.getContext();
            eventPublisher.publish(new BlueprintPageCreateEvent(this, createdPage, blueprint, user, emptyContext));
        }

        return createdContent;
    }

    @Override
    public BlueprintPage createPage(final CreateBlueprintPageEntity entity, final ConfluenceUser creator) throws BlueprintIllegalArgumentException {
        CreateBlueprintPageRequest createRequest = requestResolver.resolve(entity, creator);
        ContentBlueprint blueprint = createRequest.getContentBlueprint();

        validateBlueprintCreateResult(blueprint);
        validatePageTitleIsUnique(createRequest);

        // CONFDEV-23083: if index page is disable, do not check if bp page name duplicates index page name
        if (!blueprint.isIndexDisabled()) {
            validatePageTitleDifferentToItsIndexPage(createRequest);
        }

        final Page blueprintPage = contentGenerator.generateBlueprintPageObject(createRequest);
        addLabels(createRequest, blueprintPage);

        final Page indexPage = linkWithParentOrIndexPage(createRequest, blueprint, blueprintPage);

        pageManager.saveContentEntity(blueprintPage, DefaultSaveContext.DEFAULT);

        addPermissions(createRequest, blueprintPage);

        eventPublisher.publish(new BlueprintPageCreateEvent(this, blueprintPage, blueprint, creator,
                createRequest.getContext()));

        return new BlueprintPage(blueprintPage, indexPage);
    }

    @Override
    public Draft createDraft(CreateBlueprintPageEntity entity, ConfluenceUser creator)
            throws BlueprintIllegalArgumentException {
        if (draftsTransitionHelper.isSharedDraftsFeatureEnabled(entity.getSpaceKey())) {
            throw new UnsupportedOperationException("Cannot create legacy drafts with shared-drafts dark feature enabled. Please use createContentDraft instead.");
        }

        return (Draft) createContentDraft(entity, creator);
    }

    @Override
    public ContentEntityObject createContentDraft(CreateBlueprintPageEntity entity, ConfluenceUser creator) throws BlueprintIllegalArgumentException {
        CreateBlueprintPageRequest createRequest = requestResolver.resolve(entity, creator);

        final Page blueprintPage = contentGenerator.generateBlueprintPageObject(createRequest);

        ContentEntityObject contentDraft = saveContentDraft(createRequest, blueprintPage);
        addLabels(createRequest, contentDraft);
        addPermissions(createRequest, contentDraft);

        requestStorage.storeCreateRequest(entity, contentDraft);

        return contentDraft;
    }

    @Override
    public void deleteContentBlueprintsForSpace(@Nonnull String spaceKey) {
        // 1. Find all content blueprints created for the space
        List<ContentBlueprint> contentBlueprints = contentBlueprintManager.getAllBySpaceKey(spaceKey);
        contentBlueprints.forEach(contentBlueprint -> {
            ContentTemplateRef indexPageTemplateRef = contentBlueprint.getIndexPageTemplateRef();
            // Delete the page template associated with this index page template ref
            removePageTemplateSilently(indexPageTemplateRef.getTemplateId());
            // Delete the index pate template ref
            contentTemplateRefManager.delete(indexPageTemplateRef.getId());
            contentBlueprint.getContentTemplateRefs().forEach(contentTemplateRef -> {
                // For each content template ref entry associated with this content blueprint, we delete the page template associated with it and then delete it.
                removePageTemplateSilently(contentTemplateRef.getTemplateId());
                // Delete the content template ref
                contentTemplateRefManager.delete(contentTemplateRef.getId());
            });
            // Delete the content blueprint
            contentBlueprintManager.delete(contentBlueprint.getId());
        });
    }

    private void addPermissions(CreateBlueprintPageRequest createRequest, ContentEntityObject ceo) {
        // TODO - could expose PagePermissionData object from confluence-editor artifact?
        String viewPermissionsUsers = createRequest.getViewPermissionsUsers();
        if (isNotBlank(viewPermissionsUsers)) {
            PagePermissionsActionHelper permissionHelper = new PagePermissionsActionHelper(createRequest.getCreator()
                    , userAccessor);
            List<ContentPermission> viewPermissions = permissionHelper.createPermissions(ContentPermission.VIEW_PERMISSION, null, viewPermissionsUsers);
            contentPermissionManager.setContentPermissions(viewPermissions, ceo, ContentPermission.VIEW_PERMISSION);
        }
    }

    private void addLabels(CreateBlueprintPageRequest createRequest, Labelable labelable) {
        String labelsString = (String) createRequest.getContext().get(ContentBlueprintService.LABELS);
        if (isNotBlank(labelsString)) {
            String[] labels = labelsString.split(" ");  // can't have spaces in labels, oh no.
            for (String label : labels) {
                labelManager.addLabel(labelable, new Label(label));
            }
        }
        labelManager.addLabel(labelable, new Label(createRequest.getContentBlueprint().getIndexKey()));
    }

    @VisibleForTesting
    ContentEntityObject saveContentDraft(CreateBlueprintPageRequest createRequest, Page blueprintPage) {
        ContentEntityObject contentDraft = createRequest.getParentPage() != null
                ? draftsTransitionHelper.createDraft(Page.CONTENT_TYPE, createRequest.getSpace().getKey(), createRequest.getParentPage().getId())
                : draftsTransitionHelper.createDraft(Page.CONTENT_TYPE, createRequest.getSpace().getKey());

        contentDraft.setTitle(blueprintPage.getTitle());
        contentDraft.setBodyAsString(blueprintPage.getBodyAsString());
        if (DraftsTransitionHelper.isLegacyDraft(contentDraft)) {
            draftManager.saveDraft((Draft) contentDraft);
        } else {
            pageManager.saveContentEntity(contentDraft, DefaultSaveContext.DRAFT);
        }

        return contentDraft;
    }

    private Page linkWithParentOrIndexPage(CreateBlueprintPageRequest createRequest, ContentBlueprint blueprint, Page blueprintPage) {
        final Page indexPage = legacyManager.createAndPinIndexPage(blueprint, createRequest.getSpace
                ());
        Page parentPage = createRequest.getParentPage();
        if (parentPage == null) {
            // If the Blueprint page is not created from another page, it will be added under the Index page.
            parentPage = indexPage;
        }
        if (parentPage != null)
            parentPage.addChild(blueprintPage);
        return indexPage;
    }

    private void validatePageTitleDifferentToItsIndexPage(CreateBlueprintPageRequest createRequest)
            throws BlueprintIllegalArgumentException {
        String title = createRequest.getTitle();
        final String indexPageTitle = legacyManager.getIndexPageTitle(createRequest.getContentBlueprint());
        if (indexPageTitle.equalsIgnoreCase(title)) {
            throw new BlueprintIllegalArgumentException("Attempted to create a page with the same title as this Blueprint's Index page.", ResourceErrorType.DUPLICATED_TITLE_INDEX, title);
        }
    }

    private void validatePageTitleIsUnique(CreateBlueprintPageRequest createRequest)
            throws BlueprintIllegalArgumentException {
        // Check page title is not duplicate. Note that the title is expected to be validated with JS, so this message
        // is not i18ned.
        final String title = createRequest.getTitle();
        if (pageManager.getPage(createRequest.getSpace().getKey(), title) != null) {
            throw new BlueprintIllegalArgumentException("Attempted to create a page with the same title as an existing page.", ResourceErrorType.DUPLICATED_TITLE, title);
        }
    }

    private void validateBlueprintCreateResult(ContentBlueprint blueprint) throws BlueprintIllegalArgumentException {
        // createPage calls are only valid for "view" result blueprints. Blueprints needing to go to the editor should
        // not be created via this method.
        final String createResult = blueprint.getCreateResult();
        if (!CREATE_RESULT_VIEW.equals(createResult)) {
            throw new BlueprintIllegalArgumentException("Attempted to create a page that needs to go to the editor.", ResourceErrorType.INVALID_CREATE_RESULT_BLUEPRINT, createResult);
        }
    }

    private ContentEntityObject getDraft(ContentId draftId) {
        ContentEntityObject content = contentEntityManager.getById(draftId.asLong());
        return (content != null && content.isDraft()) ? content : null;
    }

    private Content createIndexPageAndSetAsParent(Content content, ContentBlueprint blueprint) {
        Space space = spaceManager.getSpace(content.getSpace().getKey());
        Page indexPage = legacyManager.createAndPinIndexPage(blueprint, space);

        // If the content does not have parent content and the index page exists,
        // set the index page as the parent.
        if (content.getAncestors().isEmpty() && indexPage != null) {
            Content parentContent = Content.builder(ContentType.PAGE, indexPage.getId()).build();
            content = Content.builder(content).parent(parentContent).build();
        }

        return content;
    }

    private ContentBlueprint getBlueprint(CreateBlueprintPageEntity createBlueprintPageEntity) {
        if (createBlueprintPageEntity == null)
            return null;

        if (isNotBlank(createBlueprintPageEntity.getContentBlueprintId()))
            return contentBlueprintManager.getById(UUID.fromString(createBlueprintPageEntity.getContentBlueprintId()));

        // Coming in from somewhere external that doesn't have the UUID.
        ModuleCompleteKey moduleCompleteKey = new ModuleCompleteKey(createBlueprintPageEntity.getModuleCompleteKey());
        return contentBlueprintManager.getPluginBackedContentBlueprint(moduleCompleteKey, createBlueprintPageEntity
                .getSpaceKey());
    }

    private CreateBlueprintPageEntity getCreateBlueprintPageEntity(ContentEntityObject draft) {
        CreateBlueprintPageEntity createBlueprintPageEntity;
        try {
            createBlueprintPageEntity = requestStorage.retrieveRequest(draft);
        } catch (IllegalStateException ex) { //If the draft does not contain blueprint json request, request storage will throw IllegalStateException
            createBlueprintPageEntity = null;
        }
        return createBlueprintPageEntity;
    }

    private String decorateReturnedUrlWithFlashScopeInfo(ContentBlueprint blueprint, ConfluenceUser user, String baseUrl) {
        userBlueprintConfigManager.setBlueprintCreatedByUser(blueprint.getId(), user);
        FlashScope.put(BlueprintManager.FIRST_BLUEPRINT_FOR_USER, blueprint.getId());
        FlashScope.put(BlueprintConstants.INDEX_DISABLED, blueprint.isIndexDisabled());

        String flashId = FlashScope.persist();
        return FlashScope.getFlashScopeUrl(baseUrl, flashId);
    }

    private void removePageTemplateSilently(long pageTemplateId) {
        if (pageTemplateId == 0) {
            return;
        }
        PageTemplate pageTemplate = pageTemplateManager.getPageTemplate(pageTemplateId);
        if (pageTemplate != null) {
            pageTemplateManager.removePageTemplate(pageTemplate);
        }
    }
}
