package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintCleaner;
import com.atlassian.confluence.plugins.createcontent.activeobjects.ContentBlueprintAo;
import com.atlassian.confluence.plugins.createcontent.api.services.ContentBlueprintService;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.spaces.SpacesQuery;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import net.java.ao.Query;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.Set;
import java.util.stream.Collectors;

@Component
public class DefaultContentBlueprintCleaner implements ContentBlueprintCleaner {
    private static final Logger log = LoggerFactory.getLogger(DefaultContentBlueprintCleaner.class);
    private final TransactionTemplate transactionTemplate;
    private final ContentBlueprintService contentBlueprintService;
    private final ActiveObjects activeObjects;
    private final SpaceManager spaceManager;

    @Autowired
    public DefaultContentBlueprintCleaner(
            final @ComponentImport TransactionTemplate transactionTemplate,
            final ContentBlueprintService contentBlueprintService,
            final @ComponentImport ActiveObjects activeObjects,
            final @ComponentImport SpaceManager spaceManager) {
        this.transactionTemplate = transactionTemplate;
        this.contentBlueprintService = contentBlueprintService;
        this.activeObjects = activeObjects;
        this.spaceManager = spaceManager;
    }

    @Override
    public int cleanUp() {
        try {
            // Get all removed spaces associated with any content blueprints
            Set<String> spaceKeys = getAllRemovedSpaceKeys();
            return transactionTemplate.execute(() -> {
                spaceKeys.forEach(contentBlueprintService::deleteContentBlueprintsForSpace);
                return spaceKeys.size();
            });
        } catch (Exception e) {
            log.error("Could not clean up content blueprints for removed space: {}", e);
            return -1;
        }
    }

    @VisibleForTesting
    Set<String> getAllRemovedSpaceKeys() {
        Query query = Query.select();
        query.setWhereClause("SPACE_KEY IS NOT NULL");
        ContentBlueprintAo[] aos = activeObjects.find(ContentBlueprintAo.class, query);
        Set<String> spaceKeys = Arrays.stream(aos)
                .map(ContentBlueprintAo::getSpaceKey)
                .collect(Collectors.toSet());
        // Get the list of existing space keys
        SpacesQuery spacesQuery = SpacesQuery.newQuery().withSpaceKeys(spaceKeys).build();
        Set<String> existingSpaceKeys = spaceManager.getAllSpaces(spacesQuery)
                .stream()
                .map(Space::getKey)
                .collect(Collectors.toSet());
        spaceKeys.removeAll(existingSpaceKeys);
        return spaceKeys;
    }
}
