package com.atlassian.confluence.plugins.createcontent.api.contextproviders;

import com.atlassian.annotations.PublicApi;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.web.ContextProvider;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.ANALYTICS_KEY;
import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.BLUEPRINT_ID;
import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.BLUEPRINT_MODULE_KEY;
import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.CONTENT_PAGE_TITLE;
import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.CREATE_FROM_TEMPLATE_LABEL;
import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.CREATE_RESULT;
import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.SPACE_KEY;
import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.TEMPLATE_LABEL;

/**
 * Wraps the Blueprint Context map to provide getters and setters rather than direct access to the map values.
 */
@PublicApi
public class BlueprintContext {
    private final Map<String, Object> map;

    /**
     * This method is package-level on purpose: only the {@link AbstractBlueprintContextProvider} should call it. Other classes
     * should interact with the map via the facade methods.
     *
     * @param context a map for use with the {@link ContextProvider} interface
     */
    BlueprintContext(Map<String, Object> context) {
        this.map = context;
    }

    public BlueprintContext() {
        this(new HashMap<>());
    }

    /**
     * This method is note for use in plugins: only the {@link AbstractBlueprintContextProvider} should call it. Other classes
     * should interact with the map via the facade methods.
     *
     * @return a map for use with the {@link ContextProvider} interface
     */
    public Map<String, Object> getMap() {
        return map;
    }

    public ModuleCompleteKey getBlueprintModuleCompleteKey() {
        return (ModuleCompleteKey) get(BLUEPRINT_MODULE_KEY);
    }

    public UUID getBlueprintId() {
        return (UUID) get(BLUEPRINT_ID);
    }

    public String getSpaceKey() {
        return (String) get(SPACE_KEY);
    }

    public String getAnalyticsKey() {
        return (String) get(ANALYTICS_KEY);
    }

    public String getTemplateLabel() {
        return (String) get(TEMPLATE_LABEL);
    }

    public String getCreateResult() {
        return (String) get(CREATE_RESULT);
    }

    public String getCreateFromTemplateLabel() {
        return (String) get(CREATE_FROM_TEMPLATE_LABEL);
    }

    /**
     * Use this get method sparingly!
     */
    public Object get(String key) {
        return map.get(key);
    }

    /**
     * Use this put method sparingly!
     */
    public void put(String key, Object value) {
        map.put(key, value);
    }

    /**
     * Set the title that the content (page, blog) should have.
     */
    public void setTitle(String title) {
        put(CONTENT_PAGE_TITLE, title);
    }

    /**
     * Set the label for a Create-from-Template macro button that may be rendered from this context.
     */
    public void setCreateFromTemplateLabel(String label) {
        put(CREATE_FROM_TEMPLATE_LABEL, label);

        // For short-term backwards-compatibility with plugins, we need to expose the "createFromTemplateLabel" value as
        // "pageFromTemplateTitle". This code should be deleted by, mmm, let's say 2015.
        put("pageFromTemplateTitle", label);
    }

    private Object get(BlueprintContextKeys contextKey) {
        return map.get(contextKey.key());
    }

    private void put(BlueprintContextKeys contextKey, Object value) {
        map.put(contextKey.key(), value);
    }

    public void setTemplateLabel(String label) {
        put(TEMPLATE_LABEL, label);
    }

    public void setAnalyticsKey(String key) {
        put(ANALYTICS_KEY, key);
    }

    public void setSpaceKey(String spaceKey) {
        put(SPACE_KEY, spaceKey);
    }

    public void setBlueprintModuleCompleteKey(ModuleCompleteKey key) {
        put(BLUEPRINT_MODULE_KEY, key);
    }

    public void setBlueprintId(UUID blueprintId) {
        put(BLUEPRINT_ID, blueprintId);
    }
}
