package com.atlassian.confluence.plugins.content_report;

import com.atlassian.bonnie.Searchable;
import com.atlassian.confluence.content.render.xhtml.ConversionContext;
import com.atlassian.confluence.core.ContextPathHolder;
import com.atlassian.confluence.core.DateFormatter;
import com.atlassian.confluence.core.FormatSettingsManager;
import com.atlassian.confluence.core.datetime.FriendlyDateFormatter;
import com.atlassian.confluence.languages.LocaleManager;
import com.atlassian.confluence.like.LikeManager;
import com.atlassian.confluence.macro.Macro;
import com.atlassian.confluence.macro.MacroExecutionException;
import com.atlassian.confluence.macro.query.BooleanQueryFactory;
import com.atlassian.confluence.pages.AbstractPage;
import com.atlassian.confluence.pages.CommentManager;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.renderer.template.TemplateRenderer;
import com.atlassian.confluence.search.v2.ContentSearch;
import com.atlassian.confluence.search.v2.ISearch;
import com.atlassian.confluence.search.v2.InvalidSearchException;
import com.atlassian.confluence.search.v2.SearchFilter;
import com.atlassian.confluence.search.v2.SearchManager;
import com.atlassian.confluence.search.v2.SearchQuery;
import com.atlassian.confluence.search.v2.filter.SubsetResultFilter;
import com.atlassian.confluence.search.v2.query.BooleanQuery;
import com.atlassian.confluence.search.v2.query.ContentTypeQuery;
import com.atlassian.confluence.search.v2.query.InSpaceQuery;
import com.atlassian.confluence.search.v2.query.LabelQuery;
import com.atlassian.confluence.search.v2.searchfilter.ContentPermissionsSearchFilter;
import com.atlassian.confluence.search.v2.searchfilter.SpacePermissionsSearchFilter;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.setup.settings.SettingsManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUserPreferences;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.confluence.web.UrlBuilder;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.user.User;
import com.atlassian.user.impl.DefaultUser;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.atlassian.confluence.core.datetime.RequestTimeThreadLocal.getTimeOrNow;
import static com.atlassian.confluence.macro.Macro.BodyType.NONE;
import static com.atlassian.confluence.macro.Macro.OutputType.BLOCK;
import static com.atlassian.confluence.search.service.ContentTypeEnum.BLOG;
import static com.atlassian.confluence.search.service.ContentTypeEnum.PAGE;
import static com.atlassian.confluence.search.v2.SearchManager.EntityVersionPolicy.LATEST_VERSION;
import static com.atlassian.confluence.search.v2.sort.ModifiedSort.DESCENDING;
import static com.atlassian.confluence.security.Permission.VIEW;
import static com.google.common.collect.Lists.newLinkedList;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static java.util.Collections.emptySet;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.join;
import static org.apache.commons.lang3.StringUtils.trim;

public class ContentReportTableMacro implements Macro {
    private static final String TEMPLATE_PROVIDER_PLUGIN_KEY = "com.atlassian.confluence.plugins.confluence-content-report-plugin:resources";

    private static final String TEMPLATE_NAME = "Confluence.Templates.Plugins.ContentReport.contentReportTable.soy";

    private static final int MAX_RESULTS = 20;

    private static final String DO_SEARCH_URL = "/dosearchsite.action?queryString=";

    private static final String PARAM_SHOW_COMMENTS_COUNT = "showCommentsCount";

    private static final String PARAM_SHOW_LIKES_COUNT = "showLikesCount";

    private final TemplateRenderer templateRenderer;

    private final SearchManager searchManager;

    private final UserAccessor userAccessor;

    private final I18NBeanFactory i18NBeanFactory;

    private final LocaleManager localeManager;

    private final FormatSettingsManager formatSettingsManager;

    private final ContextPathHolder contextPathHolder;

    private final PermissionManager permissionManager;

    private final SpaceManager spaceManager;

    private final LikeManager likeManager;
    private final CommentManager commentManager;

    private final SettingsManager settingsManager;

    public ContentReportTableMacro(TemplateRenderer templateRenderer, SearchManager searchManager,
                                   UserAccessor userAccessor, I18NBeanFactory i18NBeanFactory, LocaleManager localeManager,
                                   FormatSettingsManager formatSettingsManager, ContextPathHolder contextPathHolder,
                                   PermissionManager permissionManager, SpaceManager spaceManager, LikeManager likeManager,
                                   CommentManager commentManager, SettingsManager settingsManager) {
        this.templateRenderer = templateRenderer;
        this.searchManager = searchManager;
        this.userAccessor = userAccessor;
        this.i18NBeanFactory = i18NBeanFactory;
        this.localeManager = localeManager;
        this.formatSettingsManager = formatSettingsManager;
        this.contextPathHolder = contextPathHolder;
        this.permissionManager = permissionManager;
        this.spaceManager = spaceManager;
        this.likeManager = likeManager;
        this.commentManager = commentManager;
        this.settingsManager = settingsManager;
    }

    @Override
    public String execute(Map<String, String> macroParameters, String ignoredBody, ConversionContext conversionContext) throws MacroExecutionException {
        final User user = getAuthenticatedUser();
        final I18NBean i18NBean = i18NBeanFactory.getI18NBean(localeManager.getLocale(user));

        return execute(macroParameters, conversionContext, user, i18NBean, createFriendlyDateFormatter(user));
    }

    String execute(final Map<String, String> macroParameters, ConversionContext conversionContext, final User user, final I18NBean i18NBean,
                   final FriendlyDateFormatter friendlyDateFormatter) throws MacroExecutionException {
        final BooleanQuery query = createQuery(macroParameters);

        // Default value is not being passed in the macro parameters
        int maxResults = macroParameters.containsKey("maxResults") ? Integer.parseInt(macroParameters.get("maxResults")) : MAX_RESULTS;

        final List<Searchable> searchables = performSearch(query, maxResults);

        ContentReportData contentReportData = createReportData(searchables, macroParameters);
        final List<?> results = buildSearchResults(i18NBean, friendlyDateFormatter, searchables, contentReportData);

        // Link users to search page if query results reach the maximum
        return renderTemplate(macroParameters, conversionContext, user, contentReportData, results, maxResults);
    }

    private String renderTemplate(final Map<String, String> macroParameters, ConversionContext conversionContext, final User user, ContentReportData contentReportData,
                                  final List<?> results, final int maxResults) throws MacroExecutionException {
        final HashMap<String, Object> templateRenderContext = newLinkedHashMap();
        templateRenderContext.put("results", results);
        templateRenderContext.put("canViewProfiles", permissionManager.hasPermission(user, VIEW, new DefaultUser()));
        templateRenderContext.put("contextPath", contextPathHolder.getContextPath());
        templateRenderContext.put("analyticsKey", macroParameters.get("analytics-key"));
        templateRenderContext.put("showCommentsCount", contentReportData.hasCommentCounts());
        templateRenderContext.put("showLikesCount", contentReportData.hasLikeCounts());

        boolean showMoreResults = (results.size() == maxResults);
        templateRenderContext.put("showMoreResults", showMoreResults);

        if (showMoreResults) {
            String queryString = buildSearchMoreResultsLinkUrl(macroParameters);
            templateRenderContext.put("searchMoreResultsLinkUrl", queryString);
        }

        // blueprint blank experience, only applicable when there are extra parameters
        String blueprintModuleCompleteKey = macroParameters.get("blueprintModuleCompleteKey");
        if (results.isEmpty() && isNotBlank(blueprintModuleCompleteKey)) {
            templateRenderContext.put("blankTitle", macroParameters.get("blankTitle"));
            templateRenderContext.put("blankDescription", macroParameters.get("blankDescription"));

            templateRenderContext.put("blueprintKey", getModuleKey(blueprintModuleCompleteKey));

            String contentBlueprintId = macroParameters.get("contentBlueprintId");
            templateRenderContext.put("contentBlueprintId", contentBlueprintId);

            String spaceKey = conversionContext.getSpaceKey();
            templateRenderContext.put("dataSpaceKey", spaceKey);

            Space space = spaceManager.getSpace(spaceKey);
            boolean canCreate = permissionManager.hasCreatePermission(getAuthenticatedUser(), space, Page.class);
            templateRenderContext.put("createButtonLabel", canCreate ? macroParameters.get("createButtonLabel") : null);

            String createContentUrl = getCreateContentUrl(contentBlueprintId, spaceKey);
            templateRenderContext.put("createContentUrl", createContentUrl);

        }

        final StringBuilder templateBuffer = new StringBuilder();
        templateRenderer.renderTo(templateBuffer, TEMPLATE_PROVIDER_PLUGIN_KEY, TEMPLATE_NAME, templateRenderContext);

        return templateBuffer.toString();
    }

    private User getAuthenticatedUser() {
        return AuthenticatedUserThreadLocal.get();
    }

    private String buildSearchMoreResultsLinkUrl(Map<String, String> macroParameters) {
        String queryString = DO_SEARCH_URL;
        Set<String> labels = getLabels(macroParameters);
        if (!labels.isEmpty()) {
            queryString += "labelText:(" + join(labels, "+OR+") + ")";
        }

        Set<String> spaceKeys = getSpaceKeys(macroParameters);
        if (!spaceKeys.isEmpty()) {
            queryString += "+AND+spacekey:(" + join(spaceKeys, "+OR+") + ")";
        }
        queryString += "&type=page,blog";
        return queryString;
    }

    private List<Map<Object, Object>> buildSearchResults(final I18NBean i18NBean, final FriendlyDateFormatter friendlyDateFormatter,
                                                         final List<Searchable> searchables, ContentReportData contentReportData) {
        final List<Map<Object, Object>> results = newLinkedList();
        for (final Searchable searchable : searchables) {
            if (!(searchable instanceof AbstractPage)) {
                continue; // only reporting on pages and blogs for now
            }

            final AbstractPage abstractPage = (AbstractPage) searchable;
            final Map<Object, Object> result = createSearchResult(i18NBean, friendlyDateFormatter,
                    contentReportData, abstractPage);

            results.add(result);
        }
        return results;
    }

    private ImmutableMap<Object, Object> createSearchResult(final I18NBean i18NBean, final FriendlyDateFormatter friendlyDateFormatter,
                                                            ContentReportData contentReportData, final AbstractPage abstractPage) {
        final User creator = abstractPage.getCreator();

        ImmutableMap.Builder<Object, Object> builder = ImmutableMap.builder()
                .put("title", abstractPage.getTitle())
                .put("urlPath", contextPathHolder.getContextPath() + abstractPage.getUrlPath())
                .put("creatorName", creator == null ? i18NBean.getText("anonymous.name") : creator.getName())
                .put("creatorFullName", creator == null ? i18NBean.getText("anonymous.name") : creator.getFullName())
                .put("friendlyModificationDate", i18NBean
                        .getText(friendlyDateFormatter.getFormatMessage(abstractPage.getLastModificationDate())))
                .put("sortableDate", Long.toString(abstractPage.getLastModificationDate().getTime()));

        if (contentReportData.hasCommentCounts()) {
            builder.put("commentCount", contentReportData.getCommentCount(abstractPage));
        }

        if (contentReportData.hasLikeCounts()) {
            builder.put("likeCount", contentReportData.getLikeCount(abstractPage));
        }

        return builder.build();
    }

    private List<Searchable> performSearch(final SearchQuery query, final int maxResults) throws MacroExecutionException {
        final SearchFilter searchFilter = ContentPermissionsSearchFilter
                .getInstance().and(SpacePermissionsSearchFilter.getInstance());
        final ISearch search = new ContentSearch(query, DESCENDING, searchFilter, new SubsetResultFilter(maxResults));
        try {
            return searchManager.searchEntities(search, LATEST_VERSION);
        } catch (InvalidSearchException e) {
            throw new MacroExecutionException("Invalid search", e);
        }
    }

    private BooleanQuery createQuery(final Map<String, String> macroParameters) {
        final BooleanQueryFactory booleanQueryFactory = new BooleanQueryFactory();

        final Set<String> labels = getLabels(macroParameters);
        booleanQueryFactory.addMust(getLabelQuery(labels));

        booleanQueryFactory.addMust(new ContentTypeQuery(ImmutableSet.of(PAGE, BLOG)));

        final Set<String> spaceKeys = getSpaceKeys(macroParameters);
        if (!spaceKeys.isEmpty()) {
            booleanQueryFactory.addMust(new InSpaceQuery(spaceKeys));
        }

        return booleanQueryFactory.toBooleanQuery();
    }

    private Set<String> getLabels(Map<String, String> macroParameters) {
        final String labelsParameter = macroParameters.get("labels");

        return splitTrimToSet(labelsParameter, ",");
    }

    private Set<String> getSpaceKeys(Map<String, String> macroParameters) {
        final String spacesParameter = macroParameters.get("spaces");
        return splitTrimToSet(spacesParameter, ",");
    }

    private BooleanQuery getLabelQuery(final Set<String> labels) {
        BooleanQueryFactory booleanQueryFactory = new BooleanQueryFactory();

        for (String label : labels) {
            booleanQueryFactory.addShould(new LabelQuery(label));
        }
        return booleanQueryFactory.toBooleanQuery();
    }

    static Set<String> splitTrimToSet(final String str, final String delimiter) {
        if (isBlank(str)) {
            return emptySet();
        } else {
            final ImmutableSet.Builder<String> builder = ImmutableSet.builder();

            for (final String token : str.split(delimiter)) {
                final String trimmed = trim(token);
                if (isNotBlank(trimmed)) {
                    builder.add(trimmed);
                }
            }

            return builder.build();
        }
    }

    @Override
    public BodyType getBodyType() {
        return NONE;
    }

    @Override
    public OutputType getOutputType() {
        return BLOCK;
    }

    private FriendlyDateFormatter createFriendlyDateFormatter(final User user) {
        final ConfluenceUserPreferences pref = userAccessor.getConfluenceUserPreferences(user);
        final DateFormatter dateFormatter = new DateFormatter(pref.getTimeZone(), formatSettingsManager, localeManager);
        return new FriendlyDateFormatter(getTimeOrNow(), dateFormatter);
    }

    private ContentReportData createReportData(List<Searchable> searchables, Map<String, String> macroParams) {
        Map<Searchable, Integer> commentCountsMap = null;
        if (isParamEnabled(macroParams, PARAM_SHOW_COMMENTS_COUNT)) {
            commentCountsMap = commentManager.countComments(searchables);
        }

        Map<Searchable, Integer> likeCountsMap = null;
        if (isParamEnabled(macroParams, PARAM_SHOW_LIKES_COUNT)) {
            likeCountsMap = likeManager.countLikes(searchables);
        }

        return new ContentReportData(commentCountsMap, likeCountsMap);
    }

    private boolean isParamEnabled(Map<String, String> macroParams, String paramName) {
        return Boolean.parseBoolean(macroParams.get(paramName));
    }

    private String getCreateContentUrl(String contentBlueprintId, String spaceKey) {
        String baseUrl = settingsManager.getGlobalSettings().getBaseUrl();
        baseUrl += "/plugins/createcontent/createpage.action";
        UrlBuilder createContentUrl = new UrlBuilder(baseUrl);
        createContentUrl.add("spaceKey", spaceKey);
        createContentUrl.add("blueprintModuleCompleteKey", contentBlueprintId);
        return createContentUrl.toString();
    }

    private String getModuleKey(String blueprintModuleCompleteKey) {
        return (new ModuleCompleteKey(blueprintModuleCompleteKey)).getModuleKey();
    }

    private static class ContentReportData {
        private final Map<Searchable, Integer> commentCountsMap;
        private final Map<Searchable, Integer> likeCountsMap;

        public ContentReportData(Map<Searchable, Integer> commentCountsMap, Map<Searchable, Integer> likeCountsMap) {
            this.commentCountsMap = commentCountsMap;
            this.likeCountsMap = likeCountsMap;
        }

        public Integer getLikeCount(Searchable content) {
            Integer likes = likeCountsMap.get(content);
            return (likes != null) ? likes : 0;
        }

        public Integer getCommentCount(Searchable content) {
            Integer comments = commentCountsMap.get(content);
            return (comments != null) ? comments : 0;
        }

        public boolean hasLikeCounts() {
            return likeCountsMap != null;
        }

        public boolean hasCommentCounts() {
            return commentCountsMap != null;
        }
    }
}
