package com.atlassian.confluence.plugin.functest.helper;

import com.atlassian.confluence.plugin.functest.ConfluenceWebTester;
import com.atlassian.confluence.plugin.functest.remote.soap.stub.ConfluenceSoapService;
import com.atlassian.confluence.plugin.functest.remote.soap.stub.RemoteAttachment;
import com.atlassian.confluence.plugin.functest.remote.soap.stub.RemotePage;
import com.atlassian.confluence.plugin.functest.remote.soap.stub.RemotePageSummary;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import javax.xml.rpc.ServiceException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

public class PageHelper extends AbstractSpaceContentHelper implements Attachable
{

    private static final Logger LOG = Logger.getLogger(PageHelper.class);

    private long parentId;

    private String lastModifier;

    private Date lastModifiedDate;

    public PageHelper(final ConfluenceWebTester confluenceWebTester, final long id)
    {
        super(confluenceWebTester, id);
    }

    public PageHelper(final ConfluenceWebTester confluenceWebTester)
    {
        this(confluenceWebTester, 0);
    }

    public long getParentId()
    {
        return parentId;
    }

    public void setParentId(long parentId)
    {
        this.parentId = parentId;
    }

    public String getLastModifier()
    {
        return lastModifier;
    }

    public void setLastModifier(String lastModifier)
    {
        this.lastModifier = lastModifier;
    }

    public Date getLastModifiedDate()
    {
        return lastModifiedDate;
    }

    public void setLastModifiedDate(Date lastModifiedDate)
    {
        this.lastModifiedDate = lastModifiedDate;
    }

    public List getChildrenPageIds()
    {
        final RemotePageSummary[] remotePageSummaries;
        String soapSessionToken = null;
        ConfluenceSoapService confluenceSoapService;


        if (getId() > 0)
        {
            try
            {
                final List pageIds;

                soapSessionToken = confluenceWebTester.loginToSoapService();
                confluenceSoapService = confluenceWebTester.getConfluenceSoapService();

                remotePageSummaries = confluenceSoapService.getChildren(soapSessionToken, getId());
                if (null != remotePageSummaries)
                {
                    pageIds = new ArrayList(remotePageSummaries.length);

                    for (int i = 0; i < remotePageSummaries.length; ++i)
                        pageIds.add(new Long(remotePageSummaries[i].getId()));

                    return pageIds;
                }

            }
            catch (final MalformedURLException mUrlE)
            {
                LOG.error("Invalid RPC URL specified.", mUrlE);
            }
            catch (final ServiceException se)
            {
                LOG.error("Service request denied.", se);
            }
            catch (final RemoteException re)
            {
                LOG.error("There's an error in Confluence.", re);
            }
            catch (final IOException ioe)
            {
                LOG.error("Can't talk to Confluence.", ioe);
            }
            finally
            {
                confluenceWebTester.logoutFromSoapService(soapSessionToken);
            }
        }

        return null;
    }

    protected RemotePage toRemotePage()
    {
        final RemotePage remotePage = new RemotePage();

        remotePage.setId(getId());
        remotePage.setSpace(getSpaceKey());
        remotePage.setParentId(getParentId());
        remotePage.setTitle(getTitle());
        remotePage.setVersion(getVersion());
        remotePage.setContent(getContent());
        remotePage.setCreator(StringUtils.isBlank(getCreator()) ? confluenceWebTester.getCurrentUserName() : getCreator());

        if (null != getCreationDate())
        {
            final Calendar creationDate = Calendar.getInstance();

            creationDate.setTime(getCreationDate());
            remotePage.setCreated(creationDate);
        }

        remotePage.setModifier(StringUtils.isBlank(getLastModifier()) ? confluenceWebTester.getCurrentUserName() : getLastModifier());

        if (null != getLastModifiedDate())
        {
            final Calendar lastModifiedDate = Calendar.getInstance();

            lastModifiedDate.setTime(getLastModifiedDate());
            remotePage.setModified(lastModifiedDate);
        }

        return remotePage;
    }

    protected void populateHelper(final RemotePage remotePage)
    {
        setId(remotePage.getId());
        setSpaceKey(remotePage.getSpace());
        setParentId(remotePage.getParentId());
        setTitle(remotePage.getTitle());
        setVersion(remotePage.getVersion());
        setContent(remotePage.getContent());
        setCreator(remotePage.getCreator());
        setCreationDate(remotePage.getCreated().getTime());
        setLastModifier(remotePage.getModifier());
        setLastModifiedDate(remotePage.getModified().getTime());
    }

    protected boolean storeRemoteContentAndUpdateHelper(
            final String soapSessionToken,
            final ConfluenceSoapService confluenceSoapService) throws MalformedURLException, ServiceException, RemoteException
    {
        final RemotePage remotePage = confluenceSoapService.storePage(soapSessionToken, toRemotePage());

        setId(remotePage.getId());
        setVersion(remotePage.getVersion());
        return 0 < getId();
    }

    protected boolean readRemoteContentAndUpdateHelper(
            final String soapSessionToken,
            final ConfluenceSoapService confluenceSoapService) throws MalformedURLException, ServiceException, RemoteException
    {
        final RemotePage remotePage = confluenceSoapService.getPage(soapSessionToken, getId());
        final boolean success;

        if ((success = null != remotePage))
            populateHelper(remotePage);

        return success;
    }

    protected boolean deleteRemoteContent(
            final String soapSessionToken,
            final ConfluenceSoapService confluenceSoapService) throws MalformedURLException, ServiceException, RemoteException
    {
        return confluenceSoapService.removePage(soapSessionToken, getId());
    }

    public String[] getAttachmentFileNames()
    {
        String soapSessionToken = null;

        try
        {
            final ConfluenceSoapService confluenceSoapService;
            final RemoteAttachment[] remoteAttachments;
            final String[] attachmentNames;

            soapSessionToken = confluenceWebTester.loginToSoapService();
            confluenceSoapService = confluenceWebTester.getConfluenceSoapService();

            remoteAttachments = confluenceSoapService.getAttachments(soapSessionToken, getId());

            if (null != remoteAttachments)
            {
                attachmentNames = new String[remoteAttachments.length];

                for (int i = 0; i < remoteAttachments.length; ++i)
                {
                    attachmentNames[i] = remoteAttachments[i].getFileName();
                }

            }
            else
            {
                attachmentNames = new String[0];
            }

            return attachmentNames;

        }
        catch (final MalformedURLException mUrlE)
        {
            LOG.error("Invalid RPC URL specified.", mUrlE);
        }
        catch (final ServiceException se)
        {
            LOG.error("Service request denied.", se);
        }
        catch (final RemoteException re)
        {
            LOG.error("There's an error in Confluence.", re);
        }
        catch (final IOException ioe)
        {
            LOG.error("Can't talk to Confluence.", ioe);
        }
        finally
        {
            confluenceWebTester.logoutFromSoapService(soapSessionToken);
        }

        return null;
    }

    public long findBySpaceKeyAndTitle()
    {
        String soapSessionToken = null;

        try
        {
            final ConfluenceSoapService confluenceSoapService;
            final RemotePage remotePage;

            soapSessionToken = confluenceWebTester.loginToSoapService();
            confluenceSoapService = confluenceWebTester.getConfluenceSoapService();

            remotePage = confluenceSoapService.getPage(soapSessionToken, getSpaceKey(), getTitle());

            if (null != remotePage)
                return remotePage.getId();

        }
        catch (final MalformedURLException mUrlE)
        {
            LOG.error("Invalid RPC URL specified.", mUrlE);
        }
        catch (final ServiceException se)
        {
            LOG.error("Service request denied.", se);
        }
        catch (final RemoteException re)
        {
            LOG.error("There's an error in Confluence.", re);
        }
        catch (final IOException ioe)
        {
            LOG.error("Can't talk to Confluence.", ioe);
        }
        finally
        {
            confluenceWebTester.logoutFromSoapService(soapSessionToken);
        }

        return 0;
    }
}
