package com.atlassian.confluence.plugin.helper;

import com.atlassian.confluence.ConfluenceSoapService;
import com.atlassian.confluence.RemoteBlogEntry;
import com.atlassian.confluence.RemoteAttachment;
import org.apache.commons.lang.StringUtils;
import org.apache.xmlrpc.XmlRpcClient;
import org.apache.xmlrpc.XmlRpcException;

import javax.xml.rpc.ServiceException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.rmi.RemoteException;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Vector;

public class BlogPostHelper extends AbstractSpaceContentHelper implements Attachable {

    public BlogPostHelper(final long id) {
        super(id);
    }

    public BlogPostHelper() {
        this(0);
    }

    protected RemoteBlogEntry toRemoteBlogEntry() {
        final RemoteBlogEntry remoteBlogEntry = new RemoteBlogEntry();
        final Calendar creationDate = Calendar.getInstance();

        remoteBlogEntry.setId(getId());
        remoteBlogEntry.setSpace(getSpaceKey());
        remoteBlogEntry.setTitle(getTitle());
        remoteBlogEntry.setVersion(getVersion());
        remoteBlogEntry.setContent(getContent());
        remoteBlogEntry.setAuthor(StringUtils.isBlank(getCreator()) ? confluenceWebTester.getCurrentUserName() : getCreator());

        creationDate.setTime(getCreationDate());
        remoteBlogEntry.setPublishDate(creationDate);

        return remoteBlogEntry;
    }

    protected void populateHelper(final RemoteBlogEntry remoteBlogEntry) {
        setId(remoteBlogEntry.getId());
        setSpaceKey(remoteBlogEntry.getSpace());
        setTitle(remoteBlogEntry.getTitle());
        setVersion(remoteBlogEntry.getVersion());
        setContent(remoteBlogEntry.getContent());
        setCreator(remoteBlogEntry.getAuthor());
        setCreationDate(remoteBlogEntry.getPublishDate().getTime());
    }

    protected boolean storeRemoteContentAndUpdateHelper(
            final String soapSessionToken,
            final ConfluenceSoapService confluenceSoapService) throws MalformedURLException, ServiceException, RemoteException {
        final RemoteBlogEntry remoteBlogEntry = confluenceSoapService.storeBlogEntry(soapSessionToken, toRemoteBlogEntry());

        setId(remoteBlogEntry.getId());
        setVersion(remoteBlogEntry.getVersion());
        return 0 < getId();
    }

    protected boolean readRemoteContentAndUpdateHelper(
            final String soapSessionToken,
            final ConfluenceSoapService confluenceSoapService) throws MalformedURLException, ServiceException, RemoteException {
        final RemoteBlogEntry remoteBlogEntry = confluenceSoapService.getBlogEntry(soapSessionToken, getId());
        final boolean success;

        if ((success = null != remoteBlogEntry))
            populateHelper(remoteBlogEntry);

        return success;
    }

    protected boolean deleteRemoteContent(
            final String soapSessionToken,
            final ConfluenceSoapService confluenceSoapService) throws MalformedURLException, ServiceException, RemoteException {
        return confluenceSoapService.removePage(soapSessionToken, getId());
    }

    public String[] getAttachmentFileNames() {
        String soapSessionToken = null;

        try {
            final ConfluenceSoapService confluenceSoapService;
            final RemoteAttachment[] remoteAttachments;
            final String[] attachmentNames;

            soapSessionToken = confluenceWebTester.loginToSoapService();
            confluenceSoapService = confluenceWebTester.getConfluenceSoapService();

            remoteAttachments = confluenceSoapService.getAttachments(soapSessionToken, getId());

            if (null != remoteAttachments) {
                attachmentNames = new String[remoteAttachments.length];

                for (int i = 0; i < remoteAttachments.length; ++i) {
                    attachmentNames[i] = remoteAttachments[i].getFileName();
                }

            } else {
                attachmentNames = new String[0];
            }

            return attachmentNames;

        } catch (final MalformedURLException mUrlE) {
            handleInvalidSoapServiceEndpointException(mUrlE);
        } catch (final ServiceException se) {
            handleInvalidSoapServiceException(se);
        } catch (final RemoteException re) {
            handleRemoteException(re);
        } finally {
            confluenceWebTester.logoutFromSoapService(soapSessionToken);
        }

        return null;
    }

    public long findBySpaceKeyPublishedDateAndBlogPostTitle() {
        String authenticationToken = null;

        try {
            final XmlRpcClient xmlRpcClient;
            final String blogPostId;

            authenticationToken = confluenceWebTester.loginToXmlRPcService();
            xmlRpcClient = confluenceWebTester.getXmlRpcClient();

            blogPostId = (String) xmlRpcClient.execute("functest-blogpost.getBlogPostId",
                    new Vector(
                            Arrays.asList(
                                    new Object[] {
                                            authenticationToken,
                                            getSpaceKey(),
                                            getTitle(),
                                            getCreationDate()
                                    }
                            )
                    ));

            return StringUtils.isBlank(blogPostId) ? 0 : Long.parseLong(blogPostId);

        } catch (final MalformedURLException mUrlE) {
            handleInvalidXmlRpcServiceEndpointException(mUrlE);
        } catch (final XmlRpcException xre) {
            handleInvalidXmlRpcServiceException(xre);
        } catch (final IOException ioe) {
            handleIOException(ioe);
        } finally {
            confluenceWebTester.logoutFromXmlRpcService(authenticationToken);
        }

        return 0;
    }
}
