package com.atlassian.confluence.extra.flyingpdf.util;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.UUID;

import static java.util.Objects.requireNonNull;

public class ImageFileCacheUtils {
    private final static Logger log = LoggerFactory.getLogger(ImageFileCacheUtils.class);
    private static final String TEMP_EXPORT_FOLDER_PREFIX = "pdf-export";
    private final static ThreadLocal<File> threadLocal = new ThreadLocal<>();

    public static void initializeConfluenceTempExportDirectory(final Path tempDir) {
        File tempExportDirectory = createTempDirectoryInConfluenceTemp(TEMP_EXPORT_FOLDER_PREFIX, requireNonNull(tempDir).toFile());
        if (!tempExportDirectory.mkdir()) {
            log.error("Unable to create temp folder {}", tempExportDirectory);
        } else {
            log.debug("Temp directory {} for pdf export was created", tempExportDirectory.getAbsolutePath());
        }
        threadLocal.set(tempExportDirectory.getAbsoluteFile());
    }

    public static void initializeSandboxTempExportDirectory() {
        // current directory is the temp directory for sandbox process
        // it is the responsibility of Confluence to setup current directory properly
        File tempExportDirectory = createTempDirectoryInCurrentFolder(TEMP_EXPORT_FOLDER_PREFIX);
        if (!tempExportDirectory.mkdir()) {
            log.error("Unable to create temp folder {}", tempExportDirectory);
        } else {
            log.debug("Temp directory {} for pdf export was created", tempExportDirectory.getAbsolutePath());
        }
        threadLocal.set(tempExportDirectory.getAbsoluteFile());
    }

    /**
     * Removes <code>temp export directory</code> for current thread.
     */
    public static void removeTempDirectory() {
        try {
            if (threadLocal.get() != null) {
                FileUtils.deleteDirectory(threadLocal.get());
            }
        } catch (IOException e) {
            log.warn("Cannot remove temporary directory: " + threadLocal.get() );
        } finally {
            threadLocal.remove();
        }
    }

    /*
     *  Copy input stream to a a temp file inside export temp files.
     */
    public static File createTempFile(InputStream inputStream) throws IOException {
        File tempFile = createTempFile(threadLocal.get());
        FileUtils.copyInputStreamToFile(inputStream, tempFile);
        return tempFile;
    }

    /*
     * Copy an bytes arrays to file
     */
    public static File createTempFile(byte[] bytes) throws IOException {
        File tempFile = createTempFile(threadLocal.get());
        FileUtils.writeByteArrayToFile(tempFile, bytes);
        return tempFile;
    }

    private static File createTempDirectoryInConfluenceTemp(String prefix, final File tempDir) {
        String uniqueRandomFileName = UUID.randomUUID().toString();
        return new File(tempDir, prefix + uniqueRandomFileName);
    }

    private static File createTempDirectoryInCurrentFolder(String prefix) {
        String uniqueRandomFileName = UUID.randomUUID().toString();
        Path currentRelativePath = Paths.get("");
        String currentFolder = currentRelativePath.toAbsolutePath().toString();
        return new File(currentFolder, prefix + uniqueRandomFileName);
    }

    private static File createTempFile(File directory) {
        String uniqueRandomFileName = UUID.randomUUID().toString();
        return new File(directory, uniqueRandomFileName);
    }
}
