package com.atlassian.confluence.extra.flyingpdf.impl;

import com.atlassian.confluence.api.model.Depth;
import com.atlassian.confluence.api.model.Expansions;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.model.pagination.SimplePageRequest;
import com.atlassian.confluence.api.service.content.SpaceService;
import com.atlassian.confluence.core.ApiRestEntityFactory;
import com.atlassian.confluence.extra.flyingpdf.PdfExportProgressMonitor;
import com.atlassian.confluence.extra.flyingpdf.PdfExporterService;
import com.atlassian.confluence.extra.flyingpdf.analytic.ExportStatus;
import com.atlassian.confluence.extra.flyingpdf.analytic.PageExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.analytic.SpaceExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy;
import com.atlassian.confluence.extra.flyingpdf.html.LinkRenderingDetails;
import com.atlassian.confluence.extra.flyingpdf.html.XhtmlBuilder;
import com.atlassian.confluence.extra.flyingpdf.util.PdfUtils;
import com.atlassian.confluence.importexport.ImportExportException;
import com.atlassian.confluence.importexport.ImportExportManager;
import com.atlassian.confluence.pages.AbstractPage;
import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.ContentNode;
import com.atlassian.confluence.pages.ContentTree;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.util.GeneralUtil;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.core.util.ProgressMeter;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.user.User;
import com.atlassian.util.profiling.UtilTimerStack;
import org.springframework.stereotype.Component;
import org.w3c.dom.Document;

import java.io.File;
import java.io.FileNotFoundException;

/**
 * An implementation of the PdfExporterService. It should be noted that at this stage this service does no permission
 * checking so shouldn't really be considered a service.
 */
@Component
public class InternalPdfExporterService implements PdfExporterService {

    private final I18NBeanFactory i18NBeanFactory;
    private final ApiRestEntityFactory spaceFactory;
    private final SpaceService apiSpaceService;
    private final XhtmlBuilder intermediateHtmlBuilder;
    private final ImportExportManager importExportManager;
    private final ExportPermissionChecker exportPermissionChecker;
    private final FlyingSaucerXmlToPdfConverter flyingPdfDocumentConverter;

    public InternalPdfExporterService(
            @ComponentImport I18NBeanFactory i18NBeanFactory,
            @ComponentImport ApiRestEntityFactory spaceFactory,
            @ComponentImport SpaceService apiSpaceService,
            @ComponentImport ImportExportManager importExportManager,
            XhtmlBuilder intermediateHtmlBuilder,
            ExportPermissionChecker exportPermissionChecker,
            FlyingSaucerXmlToPdfConverter flyingPdfDocumentConverter) {
        this.i18NBeanFactory = i18NBeanFactory;
        this.spaceFactory = spaceFactory;
        this.apiSpaceService = apiSpaceService;
        this.intermediateHtmlBuilder = intermediateHtmlBuilder;
        this.importExportManager = importExportManager;
        this.exportPermissionChecker = exportPermissionChecker;
        this.flyingPdfDocumentConverter = flyingPdfDocumentConverter;
    }

    @Override
    public File createPdfForSpace(User user, Space space, ContentTree contentTree,
                                  PdfExportProgressMonitor progress, String contextPath,
                                  SpaceExportMetrics spaceExportMetrics, DecorationPolicy decorations) throws ImportExportException {
        exportPermissionChecker.checkAuthorization(user, space);

        UtilTimerStack.push("intermediateHtmlBuilder.buildHtml");
        Document xhtml = intermediateHtmlBuilder.buildHtml(contentTree, space, LinkRenderingDetails.anchors(), DecorationPolicy.space().combine(decorations), progress);
        UtilTimerStack.pop("intermediateHtmlBuilder.buildHtml");

        // null the potentially large content tree at this point - garbage collect if needed
        contentTree = null;

        String spaceKey = getSpaceKeyForExportFileName(space.getKey());

        progress.beginHtmlToPdfConversion();
        final File file = flyingPdfDocumentConverter.convertXhtmlToPdf(spaceKey, xhtml, progress, contextPath);

        spaceExportMetrics.getExportResults().setPdfFileSizeBytes(file.length());
        spaceExportMetrics.getExportResults().setPdfPagesTotal(PdfUtils.numberOfPages(file));

        return file;
    }

    public File createPdfForPage(User user, AbstractPage page, String contextPath, PageExportMetrics pageExportMetrics) throws ImportExportException {
        final long startTime = System.currentTimeMillis();
        try {
            pageExportMetrics.setPageId(page.getId());
            pageExportMetrics.setPageRevision(page.getConfluenceRevision().hashCode());

            final File result = doCreatePdfForPage(user, page, contextPath);
            pageExportMetrics.getExportResults().setExportStatus(ExportStatus.OK);
            pageExportMetrics.getExportResults().setPdfPagesTotal(PdfUtils.numberOfPages(result));
            pageExportMetrics.getExportResults().setPdfFileSizeBytes(result.length());
            return result;
        } finally {
            pageExportMetrics.setTimeMs((int) (System.currentTimeMillis() - startTime));
        }
    }

    private File doCreatePdfForPage(User user, AbstractPage page, String contextPath) throws ImportExportException {
        exportPermissionChecker.checkAuthorization(user, page);

        Document xhtml;
        if (page instanceof Page) {
            ContentTree tree = newContentTree();
            tree.addRootNode(new ContentNode((Page) page));

            xhtml = intermediateHtmlBuilder.buildHtml(tree, page.getSpace(), LinkRenderingDetails.anchors(), DecorationPolicy.none());
        } else if (page instanceof BlogPost) {
            xhtml = intermediateHtmlBuilder.buildHtml((BlogPost) page);
        } else {
            throw new IllegalArgumentException("Only pages and blog post are supported");
        }

        String spaceKey = getSpaceKeyForExportFileName(page.getSpaceKey());
        File pdf;
        try {
            String filename = spaceKey + "-" + ((page.getTitle() == null) ? page.getId() : page.getTitle().replaceAll("\\s", ""));
            if (!GeneralUtil.isSafeTitleForFilesystem(filename)) {
                filename = spaceKey + "-" + page.getId();
            }
            pdf = flyingPdfDocumentConverter.convertXhtmlToPdf(filename, xhtml, contextPath);
        } catch (ImportExportException e) {
            if (e.getCause() instanceof FileNotFoundException) {
                String filename = spaceKey + "-" + page.getId();
                pdf = flyingPdfDocumentConverter.convertXhtmlToPdf(filename, xhtml, contextPath);
            } else {
                throw e;
            }
        }
        return pdf;
    }

    protected ContentTree newContentTree() {
        return new ContentTree();
    }

    public ContentTree getContentTree(User user, Space space) {
        return importExportManager.getContentTree(user, space);
    }

    public boolean isPermitted(User user, AbstractPage page) {
        return exportPermissionChecker.isPermitted(user, page);
    }

    public boolean isPermitted(User user, Space space) {
        return exportPermissionChecker.isPermitted(user, space);
    }

    @Override
    public boolean exportableContentExists(Space space) {
        return space != null && apiSpaceService
                .findContent(spaceFactory.buildRestEntityFrom(space, Expansions.EMPTY).getDelegate())
                .withDepth(Depth.ROOT)
                .fetchMany(ContentType.PAGE, new SimplePageRequest(0, 10))
                .size() > 0;
    }

    @Override
    public PdfExportProgressMonitor createProgressMonitor(ProgressMeter progressMeter) {
        return new ProgressMeterWrappingProgressMonitor(i18NBeanFactory.getI18NBean(), progressMeter);
    }

    private String getSpaceKeyForExportFileName(String spaceKey) {
        if (spaceKey.startsWith("~"))
            spaceKey = spaceKey.substring(1);
        return spaceKey;
    }

}
