package com.atlassian.confluence.extra.flyingpdf.impl;

import com.atlassian.confluence.extra.flyingpdf.PdfExportProgressMonitor;
import com.atlassian.confluence.extra.flyingpdf.XmlToPdfConverter;
import com.atlassian.confluence.extra.flyingpdf.config.FontManager;
import com.atlassian.confluence.extra.flyingpdf.html.ConfluenceNamespaceHandler;
import com.atlassian.confluence.extra.flyingpdf.util.ImageFileCacheUtils;
import com.atlassian.confluence.extra.flyingpdf.util.ImageInformationURICacheUtil;
import com.atlassian.confluence.extra.flyingpdf.util.ImageTranscoderCacheUtil;
import com.atlassian.confluence.importexport.ImportExportException;
import com.atlassian.confluence.importexport.impl.ExportFileNameGenerator;
import com.atlassian.confluence.importexport.resource.DownloadResourceManager;
import com.atlassian.confluence.setup.settings.ConfluenceDirectories;
import com.atlassian.confluence.setup.settings.SettingsManager;
import com.atlassian.plugin.spring.scanner.annotation.export.ExportAsService;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugin.webresource.WebResourceIntegration;
import com.atlassian.plugin.webresource.cdn.CDNStrategy;
import com.atlassian.util.profiling.UtilTimerStack;
import com.lowagie.text.DocListener;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Element;
import com.lowagie.text.HeaderFooter;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.BaseFont;
import com.lowagie.text.pdf.PdfDocument;
import com.lowagie.text.pdf.PdfWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.FileSystemResource;
import org.springframework.stereotype.Component;
import org.w3c.dom.Document;
import org.xhtmlrenderer.pdf.ITextFontResolver;
import org.xhtmlrenderer.pdf.ITextRenderer;
import org.xhtmlrenderer.pdf.PDFCreationListener;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Optional;

import static com.atlassian.confluence.extra.flyingpdf.util.UrlUtils.getFullUrl;

@Component
@ExportAsService(XmlToPdfConverter.class)
public class FlyingSaucerXmlToPdfConverter implements XmlToPdfConverter {
    public static final String INSTALLED_FONT_FAMILY = "ConfluenceInstalledFont";

    private final static Logger log = LoggerFactory.getLogger(FlyingSaucerXmlToPdfConverter.class);

    private final SettingsManager settingsManager;
    private final WebResourceIntegration webResourceIntegration;
    private final DownloadResourceManager downloadResourceManager;
    private final ConfluenceDirectories confluenceDirectories;
    private final FontManager pdfExportFontManager;
    private final ExportFileNameGenerator pdfExportFileNameGenerator;

    public FlyingSaucerXmlToPdfConverter(
            @ComponentImport SettingsManager settingsManager,
            @ComponentImport WebResourceIntegration webResourceIntegration,
            @ComponentImport DownloadResourceManager downloadResourceManager,
            @ComponentImport ConfluenceDirectories confluenceDirectories,
            FontManager pdfExportFontManager,
            ExportFileNameGenerator pdfExportFileNameGenerator) {
        this.settingsManager = settingsManager;
        this.webResourceIntegration = webResourceIntegration;
        this.downloadResourceManager = downloadResourceManager;
        this.confluenceDirectories = confluenceDirectories;
        this.pdfExportFontManager = pdfExportFontManager;
        this.pdfExportFileNameGenerator = pdfExportFileNameGenerator;
    }

    public File convertXhtmlToPdf(String filenamePrefix, Document xhtml, String contextPath) throws ImportExportException {
        return convertXhtmlToPdf(filenamePrefix, xhtml, null, contextPath);
    }

    public File convertXhtmlToPdf(String filenamePrefix, Document xhtml, PdfExportProgressMonitor progress, String contextPath)
            throws ImportExportException {
        File exportFile;

        try {
            exportFile = pdfExportFileNameGenerator.getExportFile(filenamePrefix);
        } catch (IOException ex) {
            throw new ImportExportException("Failed to create a location and file for the PDF export.", ex);
        }

        ITextRenderer renderer = newITextRenderer();
        useCustomFontIfConfigured(renderer.getFontResolver());
        String baseUrl = getFullUrl(settingsManager.getGlobalSettings().getBaseUrl(), contextPath);
        ConfluenceExportUserAgent callback = newConfluenceExportUserAgent(renderer, baseUrl);
        renderer.getSharedContext().setUserAgentCallback(callback);
        callback.setSharedContext(renderer.getSharedContext());

        OutputStream outstr;
        try {
            outstr = new BufferedOutputStream(new FileOutputStream(exportFile));
        } catch (FileNotFoundException ex) {
            throw new ImportExportException("Failed to created the output file " + exportFile.getAbsolutePath(), ex);
        }

        ConfluenceNamespaceHandler nsh = new ConfluenceNamespaceHandler(baseUrl);
        try {
            ImageFileCacheUtils.initializeConfluenceTempExportDirectory(confluenceDirectories.getTempDirectory());
            ImageInformationURICacheUtil.initializeCache();
            ImageTranscoderCacheUtil.initializeCache();
            renderer.setDocument(xhtml, baseUrl + contextPath + "/");
            renderer.getSharedContext().setNamespaceHandler(nsh);
            UtilTimerStack.push("FlyingSaucerXmlToPdfConverter.renderer.layout");
            setPdfCreationListener(progress, renderer);
            renderer.layout();
            renderer.createPDF(outstr);
        } catch (Exception ex) {
            final String msg = "Exception while rendering the PDF document " + exportFile.getAbsolutePath();
            throw new ImportExportException(msg, ex);
        } finally {
            UtilTimerStack.pop("FlyingSaucerXmlToPdfConverter.renderer.layoutAndPaint");
            try {
                outstr.close();
                ImageFileCacheUtils.removeTempDirectory();
            } catch (IOException ex) {
                throw new ImportExportException("Could not close the export file " + exportFile.getAbsolutePath(), ex);
            }
            ImageInformationURICacheUtil.purgeCache();
            ImageTranscoderCacheUtil.purgeCache();
        }

        return exportFile;
    }

    private void setPdfCreationListener(final PdfExportProgressMonitor progress, final ITextRenderer renderer) {
        renderer.setListener(new PDFCreationListener() {
            @Override
            public void preOpen(ITextRenderer iTextRenderer) {
                PdfWriter pdfWriter = iTextRenderer.getWriter();
                setListener(pdfWriter, progress);
            }

            @Override
            public void preWrite(ITextRenderer iTextRenderer, int pageCount) {
                if (progress != null) {
                    progress.completedCalculationOfPdfPages(pageCount);
                }
            }

            @Override
            public void onClose(ITextRenderer renderer) {
            }
        });
    }

    private void setListener(final PdfWriter pdfWriter, final PdfExportProgressMonitor progress) {
        final Optional<PdfDocument> pdfDocument = getPdfDocument(pdfWriter);
        pdfDocument.ifPresent(pdfDocument1 -> pdfDocument1.addDocListener(new PageTrackerListener(progress)));
    }

    /**
     * Hacks the accessibility for PdfWriter#getPdfDocument() which is package accessible.
     *
     * @param pdfWriter - the instance of PdfWriter
     * @return - the optional holding the PdfDocument instance
     */
    private Optional<PdfDocument> getPdfDocument(final PdfWriter pdfWriter) {
        try {
            Method method = pdfWriter.getClass().getDeclaredMethod("getPdfDocument");
            method.setAccessible(true);
            return Optional.of((PdfDocument) method.invoke(pdfWriter));
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            log.warn("Failed to invoke the method getPdfDocument() with error: " + e.getMessage());
            return Optional.empty();
        }
    }

    private static class PageTrackerListener implements DocListener {
        private final PdfExportProgressMonitor progress;

        PageTrackerListener(PdfExportProgressMonitor progress) {
            this.progress = progress;
        }

        @Override
        public void open() {
            log.debug("PDF document open");
        }

        @Override
        public void close() {
            log.debug("PDF document closed");
        }

        @Override
        public boolean newPage() {
            if (progress != null) {
                progress.performingHtmlToPdfConversionForPage("");
            }
            log.debug("PDF document newPage");
            return true;
        }

        @Override
        public boolean setPageSize(Rectangle rectangle) {
            return true;
        }

        @Override
        public boolean setMargins(float v, float v1, float v2, float v3) {
            return false;
        }

        @Override
        public boolean setMarginMirroring(boolean b) {
            return true;
        }

        @Override
        public boolean setMarginMirroringTopBottom(boolean b) {
            return true;
        }

        @Override
        public void setPageCount(int i) {
            if (progress != null) {
                progress.completedCalculationOfPdfPages(i);
            }
            log.debug("PDF document setPageCount " + i);
        }

        @Override
        public void resetPageCount() {
        }

        @Override
        public void setHeader(HeaderFooter headerFooter) {

        }

        @Override
        public void resetHeader() {

        }

        @Override
        public void setFooter(HeaderFooter headerFooter) {

        }

        @Override
        public void resetFooter() {

        }

        @Override
        public boolean add(Element element) {
            return true;
        }
    }

    protected ConfluenceExportUserAgent newConfluenceExportUserAgent(ITextRenderer renderer, String baseUrl) {
        CDNStrategy cdnStrategy = webResourceIntegration.getCDNStrategy();
        final String cdnUrl = cdnStrategy != null ? cdnStrategy.transformRelativeUrl("") : null;
        return new ConfluenceExportUserAgent(renderer.getOutputDevice(), baseUrl, cdnUrl, downloadResourceManager);
    }

    protected ITextRenderer newITextRenderer() {
        return new ITextRenderer();
    }

    protected void useCustomFontIfConfigured(ITextFontResolver fontResolver) throws ImportExportException {
        FileSystemResource fontResource = pdfExportFontManager.getInstalledFont();
        if (fontResource == null)
            return;

        try {
            fontResolver.addFont(fontResource.getPath(), INSTALLED_FONT_FAMILY, BaseFont.IDENTITY_H, true, null);
        } catch (DocumentException | IOException ex) {
            throw new ImportExportException(ex);
        }
    }
}
