package com.atlassian.confluence.extra.flyingpdf.sandbox;

import com.atlassian.confluence.extra.flyingpdf.analytic.ExportStatus;
import com.atlassian.confluence.extra.flyingpdf.analytic.PageExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.config.FontManager;
import com.atlassian.confluence.extra.flyingpdf.util.ErrorMessages;
import com.atlassian.confluence.extra.flyingpdf.util.RenderedPdfFile;
import com.atlassian.confluence.importexport.ImportExportException;
import com.atlassian.confluence.importexport.impl.ExportFileNameGenerator;
import com.atlassian.confluence.setup.settings.SettingsManager;
import com.atlassian.confluence.util.sandbox.SandboxCrashedException;
import com.atlassian.confluence.util.sandbox.SandboxException;
import com.atlassian.confluence.util.sandbox.SandboxTimeoutException;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugin.webresource.WebResourceIntegration;
import com.atlassian.plugin.webresource.cdn.CDNStrategy;
import com.google.common.base.Preconditions;
import org.springframework.core.io.FileSystemResource;
import org.springframework.stereotype.Component;
import org.w3c.dom.Document;

import java.io.File;
import java.io.IOException;

import static com.atlassian.confluence.extra.flyingpdf.util.UrlUtils.getFullUrl;

/**
 * Actual sandbox user for PDF-specific stuff
 */
@Component
public class SandboxXmlToPdfConverter {

    private final ErrorMessages errorMessages;
    private final SettingsManager settingsManager;
    private final WebResourceIntegration webResourceIntegration;
    private final FontManager pdfExportFontManager;
    private final PdfExportSandbox pdfExportSandbox;
    private final ExportFileNameGenerator pdfExportFileNameGenerator;

    public SandboxXmlToPdfConverter(
            @ComponentImport SettingsManager settingsManager,
            @ComponentImport WebResourceIntegration webResourceIntegration,
            ErrorMessages errorMessages,
            FontManager pdfExportFontManager,
            PdfExportSandbox pdfExportSandbox,
            ExportFileNameGenerator pdfExportFileNameGenerator) {
        this.errorMessages = errorMessages;
        this.webResourceIntegration = webResourceIntegration;
        this.settingsManager = settingsManager;
        this.pdfExportFontManager = pdfExportFontManager;
        this.pdfExportSandbox = pdfExportSandbox;
        this.pdfExportFileNameGenerator = pdfExportFileNameGenerator;
    }

    RenderedPdfFile convertXhtmlToPdf(String filenamePrefix,
                                      String pageTitle,
                                      Document xhtml,
                                      String contextPath,
                                      String username,
                                      PageExportMetrics pageExportMetrics) throws ImportExportException {
        final SandboxPdfConversionTask task = new SandboxPdfConversionTask();
        final File exportFile;
        try {
            exportFile = pdfExportFileNameGenerator.getExportFile(filenamePrefix);
        } catch (IOException ex) {
            throw new ImportExportException("Failed to create a location and file for the PDF export.", ex);
        }

        final FileSystemResource fontResource = pdfExportFontManager.getInstalledFont();
        final String fontPath = fontResource == null ? "" : fontResource.getPath();

        final String baseUrl = getFullUrl(settingsManager.getGlobalSettings().getBaseUrl(), contextPath);

        CDNStrategy cdnStrategy = webResourceIntegration.getCDNStrategy();
        final String cdnUrl = cdnStrategy != null ? cdnStrategy.transformRelativeUrl("") : null;

        final SandboxPdfConversionRequest request = SandboxPdfConversionRequest.stringEncoded(
                baseUrl, contextPath, cdnUrl, fontPath, username, exportFile.getAbsolutePath(), xhtml);

        try {
            SandboxPdfConversionResponse response = pdfExportSandbox.execute(task, request);
            if (response.wasUnableToReadInputDocument()) {
                // Bad news, HTML of the document is probably malformed and conversion Document -> String -> Document
                // was not idempotent.
                // The best thing we can do in this case is to try to send this document to sandbox once again, but this
                // time via Java serialisation. We didn't do this before because Java serialisation is too resource heavy.
                final SandboxPdfConversionResponse responseForSerialisedDoc = pdfExportSandbox.execute(
                        task, SandboxPdfConversionRequest.serializationEncoded(
                                baseUrl, contextPath, cdnUrl, fontPath, username, exportFile.getAbsolutePath(), xhtml));
                Preconditions.checkState(!responseForSerialisedDoc.wasUnableToReadInputDocument(),
                        "We don't expect any problems reading java-serialised document");

                return RenderedPdfFile.withKnownSize(responseForSerialisedDoc.getResultingPdf(),
                        responseForSerialisedDoc.getPdfPagesNum());
            }

            return RenderedPdfFile.withKnownSize(response.getResultingPdf(), response.getPdfPagesNum());
        } catch (SandboxTimeoutException e1) {
            pageExportMetrics.getExportResults().setExportStatus(ExportStatus.SANDBOX_TIMEOUT);
            final String message = errorMessages.pageTimeoutMessage(pageTitle);
            throw new ImportExportException(message, e1);
        } catch (SandboxException e1) {
            if (e1 instanceof SandboxCrashedException) {
                pageExportMetrics.getExportResults().setExportStatus(ExportStatus.SANDBOX_CRASH);
            } else {
                pageExportMetrics.getExportResults().setExportStatus(ExportStatus.FAIL);
            }
            throw new ImportExportException(errorMessages.pageErrorMessage(pageTitle), e1);
        }
    }

}
