package com.atlassian.confluence.extra.flyingpdf.impl;

import com.atlassian.annotations.Internal;
import com.atlassian.confluence.core.service.NotAuthorizedException;
import com.atlassian.confluence.pages.AbstractPage;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.user.User;
import org.springframework.stereotype.Component;

@Internal
@Component
public class ExportPermissionChecker {
    private final PermissionManager permissionManager;

    ExportPermissionChecker(@ComponentImport PermissionManager permissionManager) {
        this.permissionManager = permissionManager;
    }

    /**
     * Internally used method that will throw the NotAuthorizedException if the user cannot perform an export of the
     * requested target.
     *
     * @param user    the user that initiated the export
     * @param target  the object to export
     * @throws NotAuthorizedException if the user is not authorized to export the supplied target.
     */
    public void checkAuthorization(User user, Object target) {
        boolean permitted = false;
        if (target instanceof Space)
            permitted = isPermitted(user, (Space) target);
        else if (target instanceof AbstractPage)
            permitted = isPermitted(user, (AbstractPage) target);

        if (!permitted) {
            String username;
            if (user == null)
                username = "anonymous";
            else
                username = user.getName();

            throw new NotAuthorizedException("The user " + username + " is not permitted to perform this export");
        }
    }

    public boolean isPermitted(User user, AbstractPage page) {
        return permissionManager.hasPermission(user, Permission.VIEW, page);
    }

    public boolean isPermitted(User user, Space space) {
        return permissionManager.isConfluenceAdministrator(user)
                || permissionManager.hasPermission(user, Permission.EXPORT, space);
    }
}
