package com.atlassian.confluence.extra.flyingpdf.util;

import com.atlassian.config.util.BootstrapUtils;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.UUID;

public class ImageFileCacheUtils {
    private final static Logger log = LoggerFactory.getLogger(ImageFileCacheUtils.class);
    private static final String TEMP_EXPORT_FOLDER_PREFIX = "pdf-export";
    private final static ThreadLocal<String> threadLocal = new ThreadLocal<>();

    public static void initializeConfluenceTempExportDirectory() {
        File tempExportDirectory = createTempDirectoryInConfluenceTemp(TEMP_EXPORT_FOLDER_PREFIX);
        if (!tempExportDirectory.mkdir()) {
            log.error("Unable to create temp folder {}", tempExportDirectory);
        } else {
            log.debug("Temp directory {} for pdf export was created", tempExportDirectory.getAbsolutePath());
        }
        threadLocal.set(tempExportDirectory.getAbsolutePath());
    }

    public static void initializeSandboxTempExportDirectory() {
        // current directory is the temp directory for sandbox process
        // it is the responsibility of Confluence to setup current directory properly
        File tempExportDirectory = createTempDirectoryInCurrentFolder(TEMP_EXPORT_FOLDER_PREFIX);
        if (!tempExportDirectory.mkdir()) {
            log.error("Unable to create temp folder {}", tempExportDirectory);
        } else {
            log.debug("Temp directory {} for pdf export was created", tempExportDirectory.getAbsolutePath());
        }
        threadLocal.set(tempExportDirectory.getAbsolutePath());
    }

    /**
     * Removes <code>temp export directory</code> for current thread.
     */
    public static void removeTempDirectory() {
        try {
            FileUtils.deleteDirectory(new File(threadLocal.get()));
        } catch (IOException e) {
            log.warn("Cannot remove temporary directory: " + threadLocal.get() );
        } finally {
            threadLocal.remove();
        }
    }

    /*
     *  Copy input stream to a a temp file inside export temp files.
     */
    public static File createTempFile(InputStream inputStream) throws IOException {
        File tempFile = createTempFile(threadLocal.get());
        FileUtils.copyInputStreamToFile(inputStream, tempFile);
        return tempFile;
    }

    /*
     * Copy an bytes arrays to file
     */
    public static File createTempFile(byte[] bytes) throws IOException {
        File tempFile = createTempFile(threadLocal.get());
        FileUtils.writeByteArrayToFile(tempFile, bytes);
        return tempFile;
    }

    private static File createTempDirectoryInConfluenceTemp(String prefix) {
        String uniqueRandomFileName = UUID.randomUUID().toString();
        return new File(BootstrapUtils.getBootstrapManager().getFilePathProperty("webwork.multipart.saveDir"), prefix + uniqueRandomFileName);
    }

    private static File createTempDirectoryInCurrentFolder(String prefix) {
        String uniqueRandomFileName = UUID.randomUUID().toString();
        Path currentRelativePath = Paths.get("");
        String currentFolder = currentRelativePath.toAbsolutePath().toString();
        return new File(currentFolder, prefix + uniqueRandomFileName);
    }

    private static File createTempFile(String directory) {
        String uniqueRandomFileName = UUID.randomUUID().toString();
        return new File(directory, uniqueRandomFileName);
    }
}
