package com.atlassian.confluence.extra.flyingpdf.impl.rpc;

import com.atlassian.confluence.extra.flyingpdf.analytic.SpaceExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.impl.DelegatingPdfExporterService;
import com.atlassian.confluence.importexport.ImportExportException;
import com.atlassian.confluence.importexport.ImportExportManager;
import com.atlassian.confluence.pages.ContentTree;
import com.atlassian.confluence.rpc.NotPermittedException;
import com.atlassian.confluence.rpc.RemoteException;
import com.atlassian.confluence.security.GateKeeper;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.setup.settings.SettingsManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.core.filters.ServletContextThreadLocal;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.atlassian.user.User;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.IOException;

@Component
public class PdfExportRpcDelegatorImpl {
    private final GateKeeper gateKeeper;
    private final SpaceManager spaceManager;
    private final SettingsManager settingsManager;
    private final PermissionManager permissionManager;
    private final ImportExportManager importExportManager;
    private final TransactionTemplate transactionTemplate;
    private final DelegatingPdfExporterService pdfExporterService;

    public PdfExportRpcDelegatorImpl(
            @ComponentImport GateKeeper gateKeeper,
            @ComponentImport SpaceManager spaceManager,
            @ComponentImport SettingsManager settingsManager,
            @ComponentImport PermissionManager permissionManager,
            @ComponentImport ImportExportManager importExportManager,
            @ComponentImport TransactionTemplate transactionTemplate,
            DelegatingPdfExporterService pdfExporterService) {
        this.gateKeeper = gateKeeper;
        this.spaceManager = spaceManager;
        this.settingsManager = settingsManager;
        this.permissionManager = permissionManager;
        this.importExportManager = importExportManager;
        this.transactionTemplate = transactionTemplate;
        this.pdfExporterService = pdfExporterService;
    }


    public String exportSpace(final String spaceKey) throws RemoteException {
        // All of this ridiculousness is because xml-rpc extensions don't get a hibernate session.
        Object obj = transactionTemplate.execute(() -> {
            Space space = spaceManager.getSpace(spaceKey);
            if (space == null) {
                return new RemoteException("Invalid spaceKey: [" + spaceKey + "]");
            }
            // Check permissions
            User user = AuthenticatedUserThreadLocal.get();

            if (!hasPermission(user, space)) {
                return new NotPermittedException("You don't have permission to export the space: " + space.getKey());
            }

            String downloadPath;
            ContentTree contentTree = importExportManager.getContentTree(user, space);

            try {
                String contextPath = ServletContextThreadLocal.getRequest().getContextPath();
                File pdfFile = pdfExporterService.createPdfForSpace(user, space, contentTree, contextPath, new SpaceExportMetrics());

                downloadPath = importExportManager.prepareDownloadPath(pdfFile.getAbsolutePath());
                gateKeeper.addKey(downloadPath, user);
            } catch (ImportExportException | IOException e) {
                return new RemoteException(e.getMessage());
            }
            return downloadPath;
        });

        if (obj instanceof RemoteException) {
            throw (RemoteException) obj;
        }

        String downloadPath = (String) obj;
        return settingsManager.getGlobalSettings().getBaseUrl() + downloadPath;
    }

    private boolean hasPermission(User user, Space space) {
        return permissionManager.hasPermission(user, Permission.EXPORT, space);
    }

}
