package com.atlassian.confluence.extra.flyingpdf.impl;

import com.atlassian.confluence.extra.flyingpdf.PdfExporterService;
import com.atlassian.confluence.extra.flyingpdf.analytic.PageExportMetrics;
import com.atlassian.confluence.importexport.ImportExportException;
import com.atlassian.confluence.pages.actions.AbstractPageAwareAction;
import com.atlassian.confluence.security.GateKeeper;
import com.atlassian.core.filters.ServletContextThreadLocal;
import com.atlassian.xwork.HttpMethod;
import com.atlassian.xwork.PermittedMethods;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

public class ExportPageAsPdfAction extends AbstractPageAwareAction {
    private static final Logger log = LoggerFactory.getLogger(ExportPageAsPdfAction.class);
    private GateKeeper gateKeeper;
    private String downloadPath;
    private PdfExporterService pdfExporterService;
    private PdfExportSemaphore pdfExportSemaphore;

    @PermittedMethods(HttpMethod.GET)
    public String execute() {
        pdfExportSemaphore.run(this::doExecute);
        return "download";
    }

    private void doExecute() {
        try {
            String contextPath = getServletRequest().getContextPath();
            File exportedDocument = pdfExporterService.createPdfForPage(getRemoteUser(), getPage(), contextPath, new PageExportMetrics());

            // in most cases these two strings will be the same, but if the path contains % or other "illegal" characters
            // the one returned to the browser (downloadPath) and the one given to the gatekeeper must be different.
            // as the gatekeeper will receive the unencoded string when the request to serve the file arrives.
            // Either way, it certainly won't contain parameters like the pdf mime type
            // CONF-18704
            String rawPath = prepareDownloadPath(exportedDocument);

            // Explicitly set the content type as pdf CONF-17946
            downloadPath = addPdfContentTypeParam(replaceBackslashes(encodePath(rawPath)));
            String gatekeeperPath = replaceBackslashes(rawPath);

            /*
              Strip the parameters before adding to the gateKeeper.
              This is because the parameters are passed separately to the gatekeeper in
              an access request, causing it to fail if we add parameters here.
             */
            gateKeeper.addKey(gatekeeperPath, getRemoteUser());
        } catch (ImportExportException | IOException e) {
            throw new RuntimeException(e);
        }
    }

    protected HttpServletRequest getServletRequest() {
        return ServletContextThreadLocal.getRequest();
    }

    @Override
    public boolean isPermitted() {
        return pdfExporterService.isPermitted(getRemoteUser(), getPage());
    }

    private String prepareDownloadPath(final File file) throws IOException {
        // The exported file is stored in the local home
        String homeDir = getBootstrapManager().getLocalHome().getCanonicalPath();
        String canonicalPath = file.getCanonicalPath();
        int homeDirIndex = canonicalPath.indexOf(homeDir);
        String urlPath = null;

        if (homeDirIndex != -1) {
            urlPath = canonicalPath.substring(homeDirIndex + homeDir.length() + 1);
        } else {
            // strip C:\ (etc), or /
            for (File root : File.listRoots()) {
                String rootPath = root.getCanonicalPath();
                int rootIndex = canonicalPath.indexOf(rootPath);
                if (rootIndex != -1) {
                    urlPath = canonicalPath.substring(rootIndex + rootPath.length());
                    break;
                }
            }
            if (urlPath == null) {
                log.warn("Path to the download [ {} ] has not been stripped of any parent directories, and may be invalid", file);
                urlPath = file.getPath();
            }
        }
        return "/download/" + urlPath;
    }

    private String addPdfContentTypeParam(String url) {
        return url + "?contentType=application/pdf";
    }

    private String replaceBackslashes(String relativeFilePath) {
        return relativeFilePath.replaceAll("\\\\", "/");
    }

    private String encodePath(String urlPath) throws UnsupportedEncodingException {
        int lastSlash = urlPath.lastIndexOf(File.separator);
        String prefix = "";
        if (lastSlash != -1) {
            prefix = urlPath.substring(0, lastSlash);
        }
        String suffix = urlPath.substring(lastSlash + 1, urlPath.length());
        String encodedSuffix = URLEncoder.encode(suffix, "UTF-8");
        if (lastSlash != -1) {
            urlPath = prefix + '/' + encodedSuffix;
        } else {
            urlPath = encodedSuffix;
        }
        return urlPath;
    }

    public void setGateKeeper(GateKeeper gateKeeper) {
        this.gateKeeper = gateKeeper;
    }

    public String getDownloadPath() {
        return downloadPath;
    }

    public void setDiagnosticsPdfExporterService(DiagnosticPdfExporterService diagnosticsPdfExporterService) {
        this.pdfExporterService = diagnosticsPdfExporterService;
    }

    public void setPdfExportSemaphore(PdfExportSemaphore pdfExportSemaphore) {
        this.pdfExportSemaphore = pdfExportSemaphore;
    }
}
