package com.atlassian.confluence.extra.flyingpdf.config;

import com.atlassian.config.util.BootstrapUtils;
import com.atlassian.core.util.FileUtils;
import org.apache.commons.io.IOUtils;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

/**
 * A FontDao that will save resource to the fonts directory in Confluence home.
 */
@Component
public class FontsDirectoryFontDao implements FontDao {
    private static final String FONTS_DIR = "fonts";

    public Resource getFont(String fontName) throws IOException {
        File fontFile = new File(getBaseFontsDir(), fontName);
        if (!fontFile.exists())
            throw new IOException("No font file could be found with the name " + fontName);

        return new FileSystemResource(fontFile);
    }

    public void saveFont(String fontName, Resource fontResource) throws IOException {
        saveFont(fontName, fontResource, true);
    }

    public void saveFont(String fontName, Resource fontResource, boolean overwrite) throws IOException {
        if (!fontResource.exists())
            throw new IOException("Attempting to install a font resource that does not exist: "
                    + fontResource.getDescription());

        File installedFontFile = new File(getBaseFontsDir(), fontName);
        InputStream istream = fontResource.getInputStream();
        try {
            FileUtils.copyFile(istream, installedFontFile, overwrite);
        } finally {
            IOUtils.closeQuietly(istream);
        }
    }

    public void removeFont(String fontName) throws IOException {
        FileSystemResource fontResource = (FileSystemResource) getFont(fontName);

        File fontFile = fontResource.getFile();
        if (!fontFile.delete()) {
            throw new IOException("Failed to remove the font file " + fontFile.getAbsolutePath());

        }
    }

    private static File getBaseFontsDir() throws IOException {
        File fontDir = new File(BootstrapUtils.getBootstrapManager().getApplicationHome() + File.separator + FONTS_DIR);
        if (!fontDir.exists() && !fontDir.mkdir())
            throw new IOException("Failed to create the font directory " + fontDir.getAbsolutePath());

        return fontDir;
    }
}
