package com.atlassian.confluence.compat.struts2.actioncontext;

import aQute.bnd.annotation.component.Component;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Locale;
import java.util.Map;

/**
 * This class lets plugins to be compatible with all supported versions of Confluence which are using ActionContext class.
 * Since 8.0 Confluence uses newer ActionContext which is struts2 variant.
 */
@Component
public class ActionContextCompatManager implements ActionContextCompat {
    private static final Logger log = LoggerFactory.getLogger(ActionContextCompatManager.class);
    private final Supplier<ActionContextCompat> delegate;

    public ActionContextCompatManager() {
        this(ActionContextCompatManager.class.getClassLoader());
    }

    public ActionContextCompatManager(ClassLoader classLoader) {
        delegate = Suppliers.memoize(() -> initialiseActionContextCompat(classLoader));
    }

    private ActionContextCompat initialiseActionContextCompat(ClassLoader classLoader) {
        ActionContextCompat internalDelegate;
        try {
            Class.forName("com.opensymphony.xwork2.ActionContext", false, classLoader);
            internalDelegate = new ActionContextStruts2AndWWCompat("com.opensymphony.xwork2.ActionContext", classLoader);
        } catch (ClassNotFoundException e) {
            log.debug("Could not find struts2 ActionContext, falling back to webwork ActionContext", e);
            try {
                internalDelegate = new ActionContextStruts2AndWWCompat("com.opensymphony.xwork.ActionContext", classLoader);
            } catch (ReflectiveOperationException ex) {
                throw new ServiceException("ActionContext couldn't be initialized.", ex);
            }
        } catch (ReflectiveOperationException e) {
            throw new ServiceException("ActionContext couldn't be initialized.", e);
        }
        return internalDelegate;
    }

    /**
     * Sets the action's application context.
     */
    public void setApplication(Map application) {
        try {
            delegate.get().setApplication(application);
        } catch (NullPointerException ex) {
            log.error("Couldn't set the Application");
        }
    }

    /**
     * Returns a Map of the ServletContext when in a servlet environment or a generic application level Map otherwise.
     */
    public Map getApplication() {
        try {
            return delegate.get().getApplication();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the Application");
        }
        return null;
    }

    /**
     * Sets the context map
     */
    public void setContextMap(Map contextMap) {
        try {
            delegate.get().setContextMap(contextMap);
        } catch (NullPointerException ex) {
            log.error("Couldn't set the ContextMap");
        }
    }

    /**
     * Gets the context map.
     */
    public Map getContextMap() {
        try {
            return delegate.get().getContextMap();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the ContextMap");
        }
        return null;
    }

    /**
     * Sets conversion errors which occurred when executing the action.
     */
    public void setConversionErrors(Map conversionErrors) {
        try {
            delegate.get().setConversionErrors(conversionErrors);
        } catch (NullPointerException ex) {
            log.error("Couldn't set the ConversionErrors");
        }
    }

    /**
     * Gets the map of conversion errors which occurred when executing the action.
     */
    public Map getConversionErrors() {
        try {
            return delegate.get().getConversionErrors();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the ConversionErrors");
        }
        return null;
    }

    /**
     * Sets the Locale for the current action.
     */
    public void setLocale(Locale locale) {
        try {
            delegate.get().setLocale(locale);
        } catch (NullPointerException ex) {
            log.error("Couldn't set the Locale");
        }
    }

    /**
     * Gets the Locale of the current action. If no locale was ever specified the platform's default locale is used.
     */
    public Locale getLocale() {
        try {
            return delegate.get().getLocale();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the Locale");
        }
        return null;
    }

    /**
     * Sets the name of the current Action in the ActionContext.
     */
    public void setName(String name) {
        try {
            delegate.get().setName(name);
        } catch (NullPointerException ex) {
            log.error("Couldn't get the Naame");
        }

    }

    /**
     * Gets the name of the current Action.
     */
    public String getName() {
        try {
            return delegate.get().getName();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the Name");
        }
        return null;
    }

    /**
     * Sets the action parameters.
     */
    public void setParameters(Map parameters) {
        try {
            delegate.get().setParameters(parameters);
        } catch (NullPointerException ex) {
            log.error("Couldn't set the Parameters");
        }
    }

    /**
     * Returns a Map of the HttpServletRequest parameters when in a servlet environment or a generic Map of parameters otherwise.
     */
    public Map getParameters() {
        try {
            return delegate.get().getParameters();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the Parameters");
        }
        return null;
    }

    /**
     * Sets a map of action session values.
     */
    public void setSession(Map session) {
        try {
            delegate.get().setSession(session);
        } catch (NullPointerException ex) {
            log.error("Couldn't set the Session");
        }
    }

    /**
     * Gets the Map of HttpSession values when in a servlet environment or a generic session map otherwise.
     */
    public Map getSession() {
        try {
            return delegate.get().getSession();
        } catch (NullPointerException ex) {
            log.error("Couldn't get the Session");
        }
        return null;
    }

    /**
     * Returns a value that is stored in the current ActionContext by doing a lookup using the value's key.
     */
    public Object get(Object key) {
        try {
            return delegate.get().get(key);
        } catch (NullPointerException ex) {
            log.error("Couldn't get the value");
        }
        return null;
    }

    /**
     * Stores a value in the current ActionContext. The value can be looked up using the key.
     */
    public void put(Object key, Object value) {
        try {
            delegate.get().put(key, value);
        } catch (NullPointerException ex) {
            log.error("Couldn't pet the value");
        }
    }
}
