package com.atlassian.configurable;

import com.atlassian.core.util.ClassLoaderUtils;
import org.apache.log4j.Logger;

import java.util.Map;

/**
 * Represents a property which gets its values derived from a {@link com.atlassian.configurable.ValuesGenerator}
 * implementation.
 */
public class ValuesGeneratorObjectConfigurationProperty extends ObjectConfigurationPropertyImpl
{
    private static Logger log = Logger.getLogger(ValuesGeneratorObjectConfigurationProperty.class);

    private final ValuesGenerator valuesGenerator;

    /**
     * Creates a new ValuesGeneratorObjectConfigurationProperty object.
     *
     * @param name         Property name
     * @param description  Property description
     * @param defaultValue Default value
     * @param type         Field type, e.g. string, long ....
     * @param valueGeneratorClass    Instance of ValuesGenerator used to retrieve a list of available choices
     */
    public ValuesGeneratorObjectConfigurationProperty(String name, String description, String defaultValue, int type, String valueGeneratorClass, String enabledConditionClass)
    {
        super(name, description, defaultValue, type, enabledConditionClass);

        //Load an instance of the class
        ValuesGenerator valuesGenerator = ValuesGenerator.NONE;
        try
        {
            valuesGenerator = (ValuesGenerator) ClassLoaderUtils.loadClass(valueGeneratorClass, ValuesGeneratorObjectConfigurationProperty.class).newInstance();
        }
        catch (Exception e)
        {
            log.error("Could not create class: " + valueGeneratorClass, e);
        }
        this.valuesGenerator = valuesGenerator;
    }

    /**
     * Overides method in super class that uses a map stored in the object to retrieve choices.
     * This function uses a {@link ValuesGenerator} class to retrieve the values.
     *
     * @param userParams used to retrieve a cut down list of choices from the {@link ValuesGenerator} class
     * @return Map of choices dependant on this users parameters.
     */
    protected Map getInternalValues(Map userParams)
    {
        return valuesGenerator.getValues(userParams);
    }
}
