package com.atlassian.configurable;

import org.apache.commons.collections.map.ListOrderedMap;

import java.util.*;

public class ObjectConfigurationImpl implements ObjectConfiguration
{
    private final ObjectDescriptor od;
    private final Map configProperties = Collections.synchronizedMap(new ListOrderedMap());
    private final Map params = Collections.synchronizedMap(new ListOrderedMap());

    /**
     * Creates a new ObjectConfigurationImpl object.
     *
     * @param id               Identifier for the Object Configuration
     * @param configProperties A Map of properties that can be configurerd for this object e.g. name, description.
     * @param od               A Class the retrieves the Description of this object based the currently configured properties.
     * @deprecated since 28 August 2007 the ID is never read, use the constructor without the ID instead.
     */
    public ObjectConfigurationImpl(String id, Map configProperties, ObjectDescriptor od)
    {
        this(configProperties, od);
    }

    /**
     * Creates a new ObjectConfigurationImpl object.
     *
     * @param configProperties A Map of properties that can be configurerd for this object e.g. name, description.
     * @param od               A Class the retrieves the Description of this object based the currently configured properties.
     */
    public ObjectConfigurationImpl(Map configProperties, ObjectDescriptor od)
    {
        this.configProperties.putAll(configProperties);
        this.od = od;
    }

    /**
     * Initialises the object with some parameters
     *
     * @param params Map of initialisation params
     */
    public void init(Map params)
    {
        this.params.clear();
        if (params != null)
        {
            this.params.putAll(params);
        }
    }

    /**
     * Retrieves the name of a property with the specified key
     *
     * @param key Key of the property
     * @return Name of the specified property
     * @throws ObjectConfigurationException
     */
    public String getFieldName(String key) throws ObjectConfigurationException
    {
        if (configProperties.containsKey(key))
        {
            return ((ObjectConfigurationProperty) configProperties.get(key)).getName();
        }
        else
        {
            throw new ObjectConfigurationException("Field with key: " + key + " does not exist.");
        }
    }

    /**
     * Retrieves the description of a property with the specified key
     *
     * @param key Key of the property
     * @return Description of the specified property
     * @throws ObjectConfigurationException
     */
    public String getFieldDescription(String key) throws ObjectConfigurationException
    {
        if (configProperties.containsKey(key))
        {
            return ((ObjectConfigurationProperty) configProperties.get(key)).getDescription();
        }
        else
        {
            throw new ObjectConfigurationException("Field with key: " + key + " does not exist.");
        }
    }

    /**
     * Retrieves the default value for property with specified key
     *
     * @param key Key of the property
     * @return Default value of the specified property
     * @throws ObjectConfigurationException
     */
    public String getFieldDefault(String key) throws ObjectConfigurationException
    {
        if (configProperties.containsKey(key))
        {
            return ((ObjectConfigurationProperty) configProperties.get(key)).getDefault();
        }
        else
        {
            throw new ObjectConfigurationException("Field with key: " + key + " does not exist.");
        }
    }

    /**
     * Retrieves the type of the property with the specified key
     *
     * @param key Key of the property
     * @return Type of the specified property
     * @throws ObjectConfigurationException
     */
    public int getFieldType(String key) throws ObjectConfigurationException
    {
        if (configProperties.containsKey(key))
        {
            return new Integer(((ObjectConfigurationProperty) configProperties.get(key)).getType()).intValue();
        }
        else
        {
            throw new ObjectConfigurationException("Field with key: " + key + " does not exist.");
        }
    }

    /**
     * Retrieves a map of available values for property with the specified key. e.g select list values
     *
     * @param key Key of the property
     * @return List valid name/value pairs for the specified property
     * @throws ObjectConfigurationException
     */
    public Map getFieldValues(String key) throws ObjectConfigurationException
    {
        if (configProperties.containsKey(key))
        {
            final ObjectConfigurationProperty objectConfigurationProperty = (ObjectConfigurationProperty) configProperties.get(key);
            objectConfigurationProperty.init(params);
            return objectConfigurationProperty;
        }
        else
        {
            throw new ObjectConfigurationException("Field with key: " + key + " does not exist.");
        }
    }

    /**
     * All the property keys for this configuration
     *
     * @return Property keys
     */
    public String[] getFieldKeys()
    {
        String[] returnValue = new String[configProperties.size()];
        int counter = 0;

        for (Iterator iterator = configProperties.keySet().iterator(); iterator.hasNext();)
        {
            String key = (String) iterator.next();

            returnValue[counter] = key;
            counter++;
        }

        return returnValue;
    }

    public String[] getEnabledFieldKeys()
    {
        List returnValue = new ArrayList();
        for (Iterator iterator = configProperties.entrySet().iterator(); iterator.hasNext();)
        {
            Map.Entry entry = (Map.Entry) iterator.next();

            if (((ObjectConfigurationProperty) entry.getValue()).isEnabled())
            {
                returnValue.add(entry.getKey());
            }
        }

        return (String[]) returnValue.toArray(new String[returnValue.size()]);
    }


    public boolean isEnabled(String key)
    {
        ObjectConfigurationProperty property = (ObjectConfigurationProperty) configProperties.get(key);
        return property != null && property.isEnabled();
    }

    /**
     * The Description of this instance of an Object Configuration
     *
     * @param params Params used to derive
     * @return Description
     */
    public String getDescription(Map params)
    {
        return od.getDescription(configProperties, params);
    }

    public boolean allFieldsHidden()
    {
        if (configProperties != null)
        {
            for (Iterator iterator = configProperties.values().iterator(); iterator.hasNext();)
            {
                ObjectConfigurationProperty ocp = (ObjectConfigurationProperty) iterator.next();
                if (ocp.getType() != ObjectConfigurationTypes.HIDDEN)
                {
                    return false;
                }
            }
        }

        return true;
    }

    public boolean isI18NValues(String key)
    {
        return configProperties.containsKey(key) && ((ObjectConfigurationProperty) configProperties.get(key)).isI18nValues();
    }
}
