package com.atlassian.cache.memory.jmx;

import com.atlassian.cache.CacheManager;
import com.atlassian.cache.CacheStatisticsKey;

import javax.annotation.Nonnull;

/**
 * JMX MBean implementation for Guava Cache
 *
 * @since v3.1
 */
public class MemoryCacheMXBeanImpl implements MemoryCacheMXBean
{
    
    private CacheManager cacheManager;
    private String cacheName;


    public MemoryCacheMXBeanImpl(@Nonnull CacheManager cacheManager, @Nonnull String cacheName)
    {
        this.cacheManager = cacheManager;
        this.cacheName = cacheName;
    }

    @Override
    public long getRequestCount()
    {
        return getProperty(CacheStatisticsKey.REQUEST_COUNT, 0L);
    }

    @Override
    public long getHitCount()
    {
        return getProperty(CacheStatisticsKey.HIT_COUNT, 0L);
    }

    @Override
    public double getHitRate()
    {
        long requestCount = getRequestCount();
        return (requestCount == 0) ? 1.0 : (double) getHitCount() / requestCount;
    }

    @Override
    public long getMissCount()
    {
        return getProperty(CacheStatisticsKey.MISS_COUNT, 0L);
    }

    @Override
    public double getMissRate()
    {
        long requestCount = getRequestCount();
        return (requestCount == 0) ? 0.0 : (double) getMissCount() / requestCount;
    }

    @Override
    public long getLoadCount()
    {
        return getProperty(CacheStatisticsKey.LOAD_COUNT, 0L);
    }

    @Override
    public long getLoadSuccessCount()
    {
        return getProperty(CacheStatisticsKey.LOAD_SUCCESS_COUNT, 0L);
    }

    @Override
    public long getLoadExceptionCount()
    {
        return getProperty(CacheStatisticsKey.LOAD_EXCEPTION_COUNT, 0L);
    }

    @Override
    public double getLoadExceptionRate()
    {
        long loadSuccessCount = getLoadSuccessCount();
        long loadExceptionCount = getLoadExceptionCount();
        long totalLoadCount = loadSuccessCount + loadExceptionCount;
        return (totalLoadCount == 0) ? 0.0 : (double) loadExceptionCount / totalLoadCount;
    }

    @Override
    public long getTotalLoadTime()
    {
        return getProperty(CacheStatisticsKey.TOTAL_LOAD_TIME, 0L);
    }

    @Override
    public double getAverageLoadPenalty()
    {
        long totalLoadCount = getLoadSuccessCount() + getLoadExceptionCount();
        return (totalLoadCount == 0) ? 0.0 : (double) getTotalLoadTime() / totalLoadCount;
    }

    @Override
    public long getEvictionCount()
    {
        return getProperty(CacheStatisticsKey.EVICTION_COUNT, 0L);
    }

    @Override
    public long getSize()
    {
        return getProperty(CacheStatisticsKey.SIZE, 0L);
    }

    private long getProperty(CacheStatisticsKey propertyName, long defaultValue)
    {
        if(cacheManager.getManagedCache(cacheName) == null
                || cacheManager.getManagedCache(cacheName).getStatistics().get(propertyName) == null)
        {
            return defaultValue;
        }
        return cacheManager.getManagedCache(cacheName).getStatistics().get(propertyName).get();
    }
}
