package com.atlassian.cache.hazelcast;

import com.atlassian.annotations.Internal;
import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.cache.Cache;
import com.atlassian.cache.CachedReference;
import com.atlassian.cache.CacheFactory;
import com.atlassian.cache.CacheLoader;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.cache.CacheSettingsDefaultsProvider;
import com.atlassian.cache.ManagedCache;
import com.atlassian.cache.Supplier;
import com.atlassian.cache.hazelcast.HazelcastAsyncHybridCache.Invalidate;
import com.atlassian.cache.impl.AbstractCacheManager;
import com.atlassian.cache.impl.ReferenceKey;
import com.atlassian.cache.impl.StrongSupplier;
import com.atlassian.cache.impl.WeakSupplier;
import com.atlassian.cache.impl.metrics.CacheManagerMetricEmitter;
import com.atlassian.cache.impl.metrics.InstrumentedCache;
import com.atlassian.cache.impl.metrics.InstrumentedCachedReference;
import com.atlassian.hazelcast.serialization.OsgiSafe;
import com.hazelcast.cluster.MembershipAdapter;
import com.hazelcast.cluster.MembershipEvent;
import com.hazelcast.config.Config;
import com.hazelcast.config.MapConfig;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.map.IMap;
import com.hazelcast.map.impl.MapContainer;
import com.hazelcast.map.impl.MapService;
import com.hazelcast.map.impl.MapServiceContext;
import com.hazelcast.map.impl.proxy.MapProxyImpl;
import com.hazelcast.map.listener.EntryAddedListener;
import com.hazelcast.map.listener.EntryUpdatedListener;
import com.hazelcast.topic.ITopic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.io.Serializable;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Objects.requireNonNull;

/**
 * Hazelcast implementation of the {@link com.atlassian.cache.CacheManager} contract
 */
@Internal
public class HazelcastCacheManager extends AbstractCacheManager
{
    private static final Logger log = LoggerFactory.getLogger(HazelcastCacheManager.class);

    protected static final String SETTINGS_MAP_NAME = LegacyPrefixedNameFactory.PREFIX + "settings";

    private final HazelcastInstance hazelcast;
    private final CacheFactory localCacheFactory;
    private final IMap<String, CacheSettings> mapSettings;
    private final UUID mapSettingsUpdatedListenerId;
    private final UUID mapSettingsAddedListenerId;
    private final UUID membershipListenerId;
    private final HazelcastNameFactory nameFactory;

    private MapServiceContext mapServiceContext;

    public HazelcastCacheManager(HazelcastInstance hazelcast,
                                 CacheFactory localCacheFactory,
                                 CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider)
    {
        this(hazelcast, localCacheFactory, cacheSettingsDefaultsProvider, new LegacyPrefixedNameFactory());
    }

    public HazelcastCacheManager(HazelcastInstance hazelcast,
                                 CacheFactory localCacheFactory,
                                 CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider,
                                 HazelcastNameFactory nameFactory)
    {
        this(hazelcast, localCacheFactory, cacheSettingsDefaultsProvider, nameFactory, new CacheManagerMetricEmitter());

    }

    @VisibleForTesting
    HazelcastCacheManager(HazelcastInstance hazelcast,
                                 CacheFactory localCacheFactory,
                                 CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider,
                                 HazelcastNameFactory nameFactory,
                                 CacheManagerMetricEmitter metricEmitter)
    {
        super(cacheSettingsDefaultsProvider, metricEmitter);

        this.hazelcast = requireNonNull(hazelcast);
        this.localCacheFactory = requireNonNull(localCacheFactory);
        this.nameFactory = requireNonNull(nameFactory);

        this.mapSettings = hazelcast.getMap(SETTINGS_MAP_NAME);

        // add listeners to be notified of remote changes to the map config so the local config can be updated too
        this.mapSettingsUpdatedListenerId = mapSettings.addEntryListener(
                (EntryUpdatedListener<String, CacheSettings>) event -> reconfigureMap(event.getKey(), event.getValue()),
                true);
        this.mapSettingsAddedListenerId = mapSettings.addEntryListener(
                (EntryAddedListener<String, CacheSettings>) event -> configureMap(event.getKey(), event.getValue()),
                true);

        // add a listener for join events to ensure that the local cache configs are updated
        this.membershipListenerId = hazelcast.getCluster().addMembershipListener(new MembershipAdapter()
        {
            @Override
            public void memberAdded(MembershipEvent membershipEvent)
            {
                maybeUpdateMapContainers();
            }
        });
    }

    public HazelcastInstance getHazelcastInstance()
    {
        return hazelcast;
    }

    /**
     * Initializes the manager. This method must be called after the bean has been created.
     */
    @PostConstruct
    public void init()
    {
        maybeUpdateMapContainers();
    }

    @Override
    protected <K, V> ManagedCache createComputingCache(@Nonnull final String name, @Nonnull final CacheSettings settings, final CacheLoader<K, V> loader)
    {
        checkSettingsAreCompatible(name, settings);

        // when a loader is provided, always create a new ManagedCache to ensure the correct loader is being used.
        // if the cache already existed, the backing values will be reused
        return cacheCreationLocks.apply(name).withLock((Supplier<ManagedCache>) () -> {
            ManagedCache cache = null;
            if (loader == null)
            {
                java.util.function.Supplier<ManagedCache> cacheSupplier = caches.get(name);
                cache = cacheSupplier == null ? null : cacheSupplier.get();
            }
            if (loader != null || cache == null)
            {
                cache = (ManagedCache) doCreateCache(name, loader, settings);
                caches.put(name, new WeakSupplier<>(cache));
            }
            return cache;
        });
    }

    @Override
    protected ManagedCache createSimpleCache(@Nonnull final String name, @Nonnull final CacheSettings settings)
    {
        checkSettingsAreCompatible(name, settings);

        ManagedCache existing = getManagedCache(name);
        if (existing != null)
        {
            return existing;
        }

        return cacheCreationLocks.apply(name).withLock((Supplier<ManagedCache>) () -> {
            if (!caches.containsKey(name))
            {
                caches.put(name, new StrongSupplier<>((ManagedCache) doCreateCache(name, null, settings)));
            }
            return caches.get(name).get();
        });
    }

    /**
     * De-registers listeners. This method must be called when the bean is no longer required.
     */
    @PreDestroy
    public void destroy()
    {
        mapSettings.removeEntryListener(mapSettingsAddedListenerId);
        mapSettings.removeEntryListener(mapSettingsUpdatedListenerId);
        hazelcast.getCluster().removeMembershipListener(membershipListenerId);
    }

    @Nonnull
    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull final String name, @Nonnull final Supplier<V> supplier,
            @Nonnull final CacheSettings settings)
    {
        final CacheSettings mergedSettings = mergeSettings(name, settings);
        checkSettingsAreCompatible(name, mergedSettings);

        return cacheCreationLocks.apply(name).withLock((Supplier<CachedReference<V>>) () -> {
            ManagedCache cache = (ManagedCache) doCreateCachedReference(name, supplier, mergedSettings);
            caches.put(name, new WeakSupplier<>(cache));
            //noinspection unchecked
            return (CachedReference<V>) cache;
        });
    }

    protected void checkSettingsAreCompatible(String name, CacheSettings settings)
    {
    }

    /**
     * Update the cache settings of an existing cache. Called by {@link com.atlassian.cache.ManagedCache} methods
     *
     * @param mapName The name of the map to update
     * @param newSettings CacheSettings object representing the new settings
     * @return true if the update succeeded, false otherwise
     */
    public boolean updateCacheSettings(@Nonnull String mapName, @Nonnull CacheSettings newSettings)
    {
        //Do nothing if the map doesn't exist
        MapConfig mapConfig = getMapConfig(mapName);
        if (mapConfig == null || !Objects.equals(mapConfig.getName(), mapName)) {
            return false;
        }

        boolean result = reconfigureMap(mapName, newSettings);

        if (result) {
            // store the config in the settings map to trigger other nodes to also configure the IMap
            mapSettings.put(mapName, asSerializable(newSettings));
        }

        return result;
    }

    protected <K, V> Cache<K, V> createAsyncHybridCache(String name, CacheLoader<K, V> loader, CacheSettings settings)
    {
        final String topicName = nameFactory.getCacheInvalidationTopicName(name);
        final ITopic<Invalidate> topic = hazelcast.getTopic(topicName);

        return InstrumentedCache.wrap(new HazelcastAsyncHybridCache<>(name, localCacheFactory, topic, loader, this, settings));
    }

    protected <V> CachedReference<V> createAsyncHybridCachedReference(String name, Supplier<V> supplier, CacheSettings settings)
    {
        final String topicName = nameFactory.getCachedReferenceInvalidationTopicName(name);
        final ITopic<ReferenceKey> topic = hazelcast.getTopic(topicName);

        return InstrumentedCachedReference.wrap(
                new HazelcastAsyncHybridCachedReference<>(name, localCacheFactory, topic, supplier, this, settings)
        );
    }

    protected <K, V> Cache<K, V> createDistributedCache(String name, CacheLoader<K, V> loader, CacheSettings settings)
    {
        final String mapName = nameFactory.getCacheIMapName(name);
        final String counterName = nameFactory.getCacheVersionCounterName(name);
        configureMap(mapName, settings);
        final IMap<K, OsgiSafe<V>> map = hazelcast.getMap(mapName);
        final CacheVersion cacheVersion = new CacheVersion(hazelcast.getCPSubsystem().getAtomicLong(counterName));

        return InstrumentedCache.wrap(new HazelcastCache<>(name, map, loader, cacheVersion, this));
    }

    protected <V> CachedReference<V> createDistributedCachedReference(String name, Supplier<V> supplier, CacheSettings settings)
    {
        // override the settings to ensure the reference is flushable and the max is set to 1000. A low value for
        // maxEntries would trigger continuous cache invalidations because of the way map eviction works in Hazelcast.
        final CacheSettings overriddenSettings = checkNotNull(settings, "settings").override(
                new CacheSettingsBuilder().flushable().maxEntries(1000).build());

        final String mapName = nameFactory.getCachedReferenceIMapName(name);
        configureMap(mapName, overriddenSettings);
        final IMap<String, OsgiSafe<V>> map = hazelcast.getMap(mapName);

        return InstrumentedCachedReference.wrap(new HazelcastCachedReference<>(name, map, supplier, this));
    }

    protected <K, V> Cache<K, V> createHybridCache(String name, CacheLoader<K, V> loader, CacheSettings settings)
    {
        final String mapName = nameFactory.getCacheIMapName(name);
        configureMap(mapName, settings);
        final IMap<K, Long> map = hazelcast.getMap(mapName);

        return InstrumentedCache.wrap(new HazelcastHybridCache<>(name, localCacheFactory, map, loader, this));
    }

    protected <V> CachedReference<V> createHybridCachedReference(String name, Supplier<V> supplier, CacheSettings settings)
    {
        final String mapName = nameFactory.getCachedReferenceIMapName(name);
        configureMap(mapName, settings);
        final IMap<ReferenceKey, Long> map = hazelcast.getMap(mapName);

        return InstrumentedCachedReference.wrap(new HazelcastHybridCachedReference<>(name, localCacheFactory, map, supplier, this));
    }

    private boolean configureMap(String mapName, CacheSettings settings)
    {
        Config config = hazelcast.getConfig();
        MapConfig mapConfig = config.findMapConfig(mapName);
        if (Objects.equals(mapConfig.getName(), mapName))
        {
            log.debug("Using existing cache configuration for cache {}", mapName);
            mapSettings.computeIfAbsent(mapName, (key) -> asSerializable(settings));
        }
        else
        {
            mapConfig = convertAndStoreMapConfig(mapName, settings, config, mapConfig);

            // store the config in the settings map to trigger other nodes to also configure the IMap
            mapSettings.putIfAbsent(mapName, asSerializable(settings));
        }
        return updateMapContainer(mapName, mapConfig);
    }

    /**
     * This method only updates the mapConfig on hazelcast instance for particular map and never updates the mapSettings
     * field with new Config. The reason is that the following method can be called as a listener callback other
     * node(other than the originating node) which reconfigures the map(eg by updating the CacheSizeEntry) by using
     * {@link com.atlassian.cache.hazelcast.HazelcastCacheManager}.updateCacheSettings() which itself ensures the
     * mapSettings field is updated with new CacheSettings.
     */
    private boolean reconfigureMap(String mapName, CacheSettings newSettings)
    {
        Config config = hazelcast.getConfig();

        MapConfig baseConfig = getMapConfig(mapName);

        MapConfig mapConfig = convertAndStoreMapConfig(mapName, newSettings, config, baseConfig);
        return updateMapContainer(mapName, mapConfig);
    }

    private MapConfig convertAndStoreMapConfig(String mapName, CacheSettings newSettings, Config config, MapConfig baseConfig)
    {
        MapConfig newConfig = new MapConfig(baseConfig);
        newConfig.setName(mapName);
        newConfig.setStatisticsEnabled(true);
        HazelcastMapConfigConfigurator.configureMapConfig(newSettings, newConfig, hazelcast.getPartitionService().getPartitions().size());
        MapConfig oldConfig = getMapConfig(mapName);
        if (oldConfig == null || !Objects.equals(oldConfig.getName(), mapName)) {
            config.addMapConfig(newConfig);
        }
        return newConfig;
    }

    private CacheSettings asSerializable(CacheSettings settings)
    {
        if (settings instanceof Serializable)
        {
            return settings;
        }

        // DefaultCacheSettings implements Serializable, make a copy
        return new CacheSettingsBuilder(settings).build();
    }

    private <K, V> Cache<K, V> doCreateCache(String name, CacheLoader<K, V> loader, CacheSettings settings)
    {
        if (settings.getLocal(false))
        {
            return localCacheFactory.getCache(name, loader, settings);
        }

        if (settings.getReplicateViaCopy(true))
        {
            return createDistributedCache(name, loader, settings);
        }

        if (settings.getReplicateAsynchronously(true))
        {
            return createAsyncHybridCache(name, loader, settings);
        }
        else
        {
            return createHybridCache(name, loader, settings);
        }
    }

    private <V> CachedReference<V> doCreateCachedReference(String name, Supplier<V> supplier, CacheSettings settings)
    {
        if (settings.getLocal(false))
        {
            return localCacheFactory.getCachedReference(name, supplier, settings);
        }

        // remote cached reference
        if (settings.getReplicateViaCopy(true))
        {
            return createDistributedCachedReference(name, supplier, settings);
        }

        if (settings.getReplicateAsynchronously(true))
        {
            return createAsyncHybridCachedReference(name, supplier, settings);
        }
        else
        {
            return createHybridCachedReference(name, supplier, settings);
        }
    }

    private MapContainer getMapContainer(@Nonnull String name)
    {
        if (mapServiceContext == null)
        {
            MapProxyImpl<String, CacheSettings> proxy = hazelcast.getDistributedObject(MapService.SERVICE_NAME, SETTINGS_MAP_NAME);
            mapServiceContext = proxy.getService().getMapServiceContext();
        }

        return mapServiceContext.getMapContainer(name);
    }

    private boolean updateMapContainer(String mapName, MapConfig config)
    {
        final MapContainer container = getMapContainer(mapName);
        if (container == null)
        {
            log.debug("Map Container not found for map {}", mapName);
            return false;
        }

        container.setMapConfig(config);
        container.initEvictor();

        return true;
    }

    private void maybeUpdateMapContainers()
    {
        for (Map.Entry<String, CacheSettings> entry : mapSettings.entrySet())
        {
            configureMap(entry.getKey(), entry.getValue());
        }
    }

    /**
     * Get the current MapConfig for the given IMap
     *
     * @param mapName the name of the map
     * @return The current mapConfig for that map if it exists, null otherwise
     */
    @Nullable
    MapConfig getMapConfig(@Nonnull String mapName)
    {
        MapContainer mapContainer = getMapContainer(mapName);
        MapConfig mapConfig = mapContainer.getMapConfig();
        if (!mapConfig.getName().equals(mapName))
        {
            return null;
        }
        else
        {
            return mapConfig;
        }
    }

    /**
     * Gets the original CacheSettings object used to configure the given map, before it was transformed into a
     * Hazelcast MapConfig
     *
     * @param mapName the name of the map
     * @return The original CacheSettings object, or null if the map doesn't exist
     */
    @Nullable
    CacheSettings getCacheSettings(@Nonnull String mapName)
    {
        return mapSettings.get(mapName);
    }
}
