package com.atlassian.cache.hazelcast;

import java.util.Random;

import javax.annotation.Nonnull;

import com.atlassian.cache.CacheSettings;

import com.hazelcast.config.EvictionConfig;
import com.hazelcast.config.EvictionPolicy;
import com.hazelcast.config.MapConfig;
import com.hazelcast.config.NearCacheConfig;
import com.hazelcast.config.NearCachePreloaderConfig;

import static com.hazelcast.config.MaxSizePolicy.PER_NODE;

/**
 * Performs the (re)configuration of Hazelcast {@link MapConfig} objects.
 *
 * @since 2.4.0
 */
class HazelcastMapConfigConfigurator
{
    // If the cache is a hybrid cache, only the entry versions are tracked in the IMap. These versions should remain
    // cached longer than the values in local caches. A multiplier is applied to the config parameters that affect
    // cache eviction to enforce this.
    static final int HYBRID_MULTIPLIER = 2;
    // see #adjustPerNodeCapacity() for details
    static final int SMALL_CACHES_CAPACITY_MULTIPLIER = 2;

    static final int NEAR_CACHE_EXPIRY_RATIO = Integer.getInteger("atlassian.cache.nearCacheExpiryRatio", 75);

    static MapConfig configureMapConfig(CacheSettings settings, MapConfig mapConfig, int partitionsCount)
    {
        boolean hybrid = !settings.getReplicateViaCopy(true);
        Integer multiplier = hybrid ? HYBRID_MULTIPLIER : 1;
        Integer maxEntries = settings.getMaxEntries();

        final NearCacheConfig nearCacheConfig = mapConfig.getNearCacheConfig() == null ?
                new NearCacheConfig() :
                copyNearCacheConfig(mapConfig.getNearCacheConfig());

        if (maxEntries != null)
        {
            // In Hazelcast 3.8 the algorithm of calculation of per-node capacity has been changed
            // (https://github.com/hazelcast/hazelcast/issues/11646)
            // Please refer to javadocs of adjustPerNodeCapacity() for details.
            final int maxSize = adjustPerNodeCapacity(mapConfig, multiplier * maxEntries, partitionsCount);

            nearCacheConfig.getEvictionConfig().setSize(maxSize);
            nearCacheConfig.getEvictionConfig().setEvictionPolicy(EvictionPolicy.LFU);
        }

        final Long expireAfterAccess = settings.getExpireAfterAccess();
        if (expireAfterAccess != null)
        {
            final int maxIdleSeconds = multiplier * roundUpToWholeSeconds(expireAfterAccess);

            mapConfig.setMaxIdleSeconds(maxIdleSeconds);

            // Near-cache hits don't reset the last-accessed timestamp on the underlying IMap entry.
            // So make the near cache have a TTL that is less than the maxIdle of the IMap:
            //
            // nearCacheTTL = (0.75 + jitter) * maxIdleSeconds
            //
            // That way the near cache entry will always expire before the IMap - calling back through
            // and refreshing the IMap idle timer. We also add some random jitter (+ or - 0.15) so all nodes don't expire
            // at the same time.

            int jitter = new Random().nextInt(30) - 15;
            int nearCacheTtl = (int) Math.floor(((NEAR_CACHE_EXPIRY_RATIO + jitter) * maxIdleSeconds) / 100.0);
            nearCacheConfig.setTimeToLiveSeconds(Math.max(1, nearCacheTtl));
        }

        final Long expireAfterWrite = settings.getExpireAfterWrite();
        if (expireAfterWrite != null)
        {
            final int timeToLiveSeconds = multiplier * roundUpToWholeSeconds(expireAfterWrite);

            mapConfig.setTimeToLiveSeconds(timeToLiveSeconds);

            nearCacheConfig.setTimeToLiveSeconds(timeToLiveSeconds);
        }

        final boolean nearCache = settings.getReplicateAsynchronously(true);
        if (nearCache)
        {
            mapConfig.setNearCacheConfig(nearCacheConfig);
        }
        else
        {
            mapConfig.setNearCacheConfig(null);
        }

        return mapConfig;
    }

    /**
     * This method creates a copy of {@link NearCacheConfig} <strong>without</strong> copying read-only objects for
     * class fields, which may contain trash from previous invocations of near cache methods (this problem comes from
     * the fact, that HZ treats map configs as a const values after cluster startup, but atlassian cache allows to
     * change them after cluster is stared)
     *
     * @param nearCacheConfig {@link NearCacheConfig} to be copied
     * @return copy of the {@link NearCacheConfig} without read-only objects
     */
    private static NearCacheConfig copyNearCacheConfig(@Nonnull  NearCacheConfig nearCacheConfig)
    {
        NearCacheConfig nearCacheConfigCopy = new NearCacheConfig(nearCacheConfig);
        EvictionConfig evictionConfigCopy = new EvictionConfig(nearCacheConfig.getEvictionConfig());
        nearCacheConfigCopy.setEvictionConfig(evictionConfigCopy);
        NearCachePreloaderConfig preloaderConfigCopy = new NearCachePreloaderConfig(nearCacheConfigCopy.getPreloaderConfig());
        nearCacheConfigCopy.setPreloaderConfig(preloaderConfigCopy);
        return nearCacheConfigCopy;
    }

    /**
     * In Hazelcast 3.8 the algorithm of calculation of per-node capacity has been changed
     * (https://github.com/hazelcast/hazelcast/issues/11646). In a nutshell, new per-node capacity calculation is now
     * based on partition size: {@code partitionSize > desiredPerNodeSize * numberOfNodes / numberOfPartitions}
     *
     * Old capacity calculation algorithm, which calculated sum of all partition sizes for the given map, which belong
     * to the current node, was replaced with simplified formula above because, according to Hazelcast devs, "partition
     * thread should interact only with partition, which belongs to this thread" :(. My guess is this calculation is
     * based on assumption that all objects are <b>evenly distributed</b> among Hazelcast partitions (which
     * goes back to the Hazelcast key hashing algorithm, but that's different story).
     *
     * This simplified calculation comes with a price:
     * <ul>
     *     <li>This formula doesn't make sense, when {@code desiredPerNodeSize < numberOfPartitions}. In this case,
     *     all objects <b>are evicted immediately after insertion</b></li>
     *     <li>This formula is based on both static ({@code desiredPerNodeSize, numberOfPartitions}) as well as
     *     dynamic ({@code numberOfNodes}) parameters, that's why it's very hard to predict precise real capacity.</li>
     * </ul>
     *
     * So the best thing we can do here is to guarantee <b>minimum capacity</b> of
     * {@link HazelcastMapConfigConfigurator#SMALL_CACHES_CAPACITY_MULTIPLIER * partitionsCount} objects per partition,
     * unless more is requested.
     *
     * @param mapConfig cache map config
     * @param desiredPerNodeSize requested per node capacity
     * @param partitionsCount number of partitions in the cluster
     * @return adjusted node capacity
     */
    private static int adjustPerNodeCapacity(MapConfig mapConfig, int desiredPerNodeSize, int partitionsCount)
    {
        int adjustedCacheSize = Math.max(SMALL_CACHES_CAPACITY_MULTIPLIER * partitionsCount, desiredPerNodeSize);
        mapConfig.setEvictionConfig(new EvictionConfig()
                .setMaxSizePolicy(PER_NODE)
                .setSize(adjustedCacheSize)
                .setEvictionPolicy(EvictionPolicy.LFU));
        return adjustedCacheSize;
    }

    private static int roundUpToWholeSeconds(final Long expireAfterAccess)
    {
        return (int) Math.ceil(expireAfterAccess / 1000d);
    }
}
