package com.atlassian.cache.compat.guava11.memory;

import com.atlassian.cache.compat.Cache;
import com.atlassian.cache.compat.CacheLoader;
import com.atlassian.cache.compat.CacheSettings;
import com.atlassian.cache.compat.CacheSettingsBuilder;
import com.atlassian.cache.compat.CachedReference;
import com.atlassian.cache.compat.GuavaAwareCacheFactory;
import com.atlassian.cache.compat.Supplier;
import com.atlassian.cache.compat.impl.AbstractCacheFactory;
import com.atlassian.cache.compat.impl.ReferenceKey;
import com.atlassian.cache.compat.impl.StrongSupplier;
import com.atlassian.cache.compat.impl.WeakSupplier;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.LoadingCache;

import javax.annotation.Nullable;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

/**
 * Maintains a mapping of name -&#62; Cache and provides factory methods for creating and getting caches.
 *
 * @since v1.0
 */
public class MemoryCacheFactory extends AbstractCacheFactory implements GuavaAwareCacheFactory
{
    @SuppressWarnings("unchecked")
    @Override
    public <V> CachedReference<V> getCachedReference(final String name,
                                                     final Supplier<V> supplier,
                                                     final CacheSettings settings)
    {
        // Force the cache settings to be flushable and a maximum size of one.
        final CacheSettings overriddenSettings = settings.override(
                new CacheSettingsBuilder().flushable().maxEntries(1).build());


        return cacheCreationLocks.apply(name).withLock(new java.util.function.Supplier<MemoryCachedReference<V>>()
        {
            @Override
            public MemoryCachedReference<V> get()
            {
                LoadingCache<ReferenceKey, V> computingCache = createCacheBuilder(overriddenSettings).build(
                        new com.google.common.cache.CacheLoader<ReferenceKey, V>()
                        {
                            @Override
                            public V load(ReferenceKey key)
                            {
                                return supplier.get();
                            }
                        });
            return MemoryCachedReference.create(computingCache);
            }
        });
    }

    @SuppressWarnings("unchecked")
    public Cache createSimpleCache(final String name, final CacheSettings settings)
    {

        Supplier<Cache> cacheSupplier = caches.get(name);
        if (cacheSupplier != null)
        {
            Cache cache = cacheSupplier.get();
            if (cache != null)
            {
                return cache;
            }
        }
        return cacheCreationLocks.apply(name).withLock(new java.util.function.Supplier<Cache>()
            {
                @Override
                public Cache get()
                {
                    if (!caches.containsKey(name))
                    {
                        final com.google.common.cache.Cache<Object, Object> simpleCache = createCacheBuilder(settings).build();
                        MemoryCache cache = MemoryCache.create(simpleCache, name, settings);
                        caches.put(name, new StrongSupplier<Cache>(cache));
                    }
                    return caches.get(name).get();
                }
            });
    }

    @SuppressWarnings("unchecked")
    public <K,V> Cache createComputingCache(final String name, final CacheSettings settings, final CacheLoader<K, V> loader)
    {
        return cacheCreationLocks.apply(name).withLock(new java.util.function.Supplier<MemoryCache<K, V>>()
        {
            @Override
            public MemoryCache<K, V> get()
            {
                LoadingCache<K, V> computingCache = createCacheBuilder(settings)
                        .build(
                                new com.google.common.cache.CacheLoader<K, V>()
                                {
                                    @Override
                                    public V load(@Nullable final K key)
                                    {
                                        return loader.load(key);
                                    }
                                });
                MemoryCache cache = MemoryCache.create(computingCache, name, settings);
                if (!caches.containsKey(name))
                {
                    caches.put(name, new WeakSupplier<Cache>(cache));
                }
                return cache;
            }
        });
    }

    private CacheBuilder<Object, Object> createCacheBuilder(CacheSettings settings)
    {
        final CacheBuilder cacheBuilder = CacheBuilder.newBuilder();
        if (null != settings.getMaxEntries())
        {
            cacheBuilder.maximumSize(settings.getMaxEntries());
        }

        if (null != settings.getExpireAfterAccess())
        {
            cacheBuilder.expireAfterAccess(settings.getExpireAfterAccess(), MILLISECONDS);
        }
        else if (null != settings.getExpireAfterWrite())
        {
            cacheBuilder.expireAfterWrite(settings.getExpireAfterWrite(), MILLISECONDS);
        }
        return cacheBuilder;
    }
}
