package com.atlassian.cache.compat.guava10.memory;

import java.util.concurrent.ConcurrentMap;

import javax.annotation.Nullable;

import com.atlassian.cache.compat.Cache;
import com.atlassian.cache.compat.CacheLoader;
import com.atlassian.cache.compat.CacheSettings;
import com.atlassian.cache.compat.CacheSettingsBuilder;
import com.atlassian.cache.compat.CachedReference;
import com.atlassian.cache.compat.GuavaAwareCacheFactory;
import com.atlassian.cache.compat.Supplier;
import com.atlassian.cache.compat.impl.AbstractCacheFactory;
import com.atlassian.cache.compat.impl.ReferenceKey;
import com.atlassian.cache.compat.impl.StrongSupplier;
import com.atlassian.cache.compat.impl.WeakSupplier;

import com.google.common.collect.MapMaker;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

/**
 * Maintains a mapping of name -&#62; Cache and provides factory methods for creating and getting caches.
 *
 * @since v1.0
 */
public class MemoryCacheFactory extends AbstractCacheFactory implements GuavaAwareCacheFactory
{
    @SuppressWarnings("unchecked")
    @Override
    public <V> CachedReference<V> getCachedReference(final String name,
            final Supplier<V> supplier,
            final CacheSettings settings)
    {
        // Force the cache settings to be flushable and a maximum size of one.
        final CacheSettings overriddenSettings = settings.override(
                new CacheSettingsBuilder().flushable().maxEntries(1).build());


        return cacheCreationLocks.apply(name).withLock(new java.util.function.Supplier<MemoryCachedReference<V>>()
            {
            @Override
            public MemoryCachedReference<V> get()
            {
                ConcurrentMap<ReferenceKey, V> computingCache = createMapMaker(overriddenSettings).makeComputingMap(
                        new com.google.common.base.Function<ReferenceKey, V>()
                        {
                            @Override
                            public V apply(@Nullable final ReferenceKey key)
                            {
                                return supplier.get();
                            }
                        });
                return MemoryCachedReference.create(computingCache);
            }
        });
    }

    @Override
    @SuppressWarnings("unchecked")
    public Cache createSimpleCache(final String name, final CacheSettings settings)
    {

        Supplier<Cache> cacheSupplier = caches.get(name);
        if (cacheSupplier != null)
        {
            Cache cache = cacheSupplier.get();
            if (cache != null)
            {
                return cache;
            }
        }
        return cacheCreationLocks.apply(name).withLock(new java.util.function.Supplier<Cache>()
            {
                @Override
                public Cache get()
                {
                    if (!caches.containsKey(name))
                    {
                        final ConcurrentMap<Object, Object> simpleCache = createMapMaker(settings).makeMap();
                        MemoryCache cache = MemoryCache.create(simpleCache, name, settings);
                        caches.put(name, new StrongSupplier<Cache>(cache));
                    }
                    return caches.get(name).get();
                }
            });
    }

    @Override
    @SuppressWarnings("unchecked")
    public <K,V> Cache createComputingCache(final String name, final CacheSettings settings, final CacheLoader<K, V> loader)
    {
        return cacheCreationLocks.apply(name).withLock(new java.util.function.Supplier<MemoryCache<K, V>>()
        {
            @Override
            public MemoryCache<K, V> get()
            {
                ConcurrentMap<K, V> computingCache = createMapMaker(settings)
                        .makeComputingMap(
                                new com.google.common.base.Function<K, V>()
                                {
                                    @Override
                                    public V apply(@Nullable final K key)
                                    {
                                        return loader.load(key);
                                    }
                                });
                MemoryCache cache = MemoryCache.create(computingCache, name, settings);
                if (!caches.containsKey(name))
                {
                    caches.put(name, new WeakSupplier<Cache>(cache));
                }
                return cache;
            }
        });
    }

    private MapMaker createMapMaker(CacheSettings settings)
    {
        final MapMaker mapMaker = new MapMaker();
        if (null != settings.getMaxEntries())
        {
            mapMaker.maximumSize(settings.getMaxEntries());
        }

        if (null != settings.getExpireAfterAccess())
        {
            mapMaker.expireAfterAccess(settings.getExpireAfterAccess(), MILLISECONDS);
        }
        else if (null != settings.getExpireAfterWrite())
        {
            mapMaker.expireAfterWrite(settings.getExpireAfterWrite(), MILLISECONDS);
        }
        return mapMaker;
    }
}
