package com.atlassian.cache.compat.guava10.memory;

import java.util.Collection;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.Nullable;

import com.atlassian.cache.compat.Cache;
import com.atlassian.cache.compat.CacheException;
import com.atlassian.cache.compat.CacheSettings;

import com.google.common.collect.ComputationException;

/**
 * A Cache that delegates Concurrent Map.
 *
 * @since v1.0
 */
class MemoryCache<K,V> implements Cache<K,V>
{
    private final String name;
    private final ConcurrentMap<K, V> internalCache;

    private MemoryCache(final ConcurrentMap<K, V> internalCache, String name)
    {
        this.internalCache = internalCache;
        this.name = name;
    }

    static <K, V> MemoryCache<K, V> create(final ConcurrentMap<K, V> internalCache, String name, CacheSettings settings)
    {
        return new MemoryCache<K, V>(internalCache, name);
    }


    @Override
    public String getName()
    {
        return name;
    }

    @Override
    public Collection<K> getKeys()
    {
        try
        {
            return internalCache.keySet();
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void put(final K key, final V value)
    {
        try
        {
            internalCache.put(key, value);
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public V get(final K key)
    {
        try
        {
            return internalCache.get(key);
        }
        catch (ComputationException e)
        {
            throw new CacheException(e.getCause());
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void remove(final K key)
    {
        try
        {
            internalCache.remove(key);
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void removeAll()
    {
        try
        {
            internalCache.clear();
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public boolean equals(@Nullable final Object other)
    {
        if (other instanceof MemoryCache)
        {
            MemoryCache otherMemoryCache = (MemoryCache) other;
            if (internalCache.equals(otherMemoryCache.internalCache))
            {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode()
    {
        return 3 + internalCache.hashCode();
    }
}
