package com.atlassian.cache.ehcache;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.cache.CacheLoader;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.cache.CacheSettingsDefaultsProvider;
import com.atlassian.cache.CachedReference;
import com.atlassian.cache.ManagedCache;
import com.atlassian.cache.ehcache.replication.EhCacheReplicatorConfigFactory;
import com.atlassian.cache.ehcache.replication.rmi.RMICacheReplicatorConfigFactory;
import com.atlassian.cache.ehcache.wrapper.NoopValueProcessor;
import com.atlassian.cache.ehcache.wrapper.ValueProcessor;
import com.atlassian.cache.ehcache.wrapper.ValueProcessorAtlassianCacheLoaderDecorator;
import com.atlassian.cache.impl.AbstractCacheManager;
import com.atlassian.cache.impl.metrics.CacheManagerMetricEmitter;
import com.atlassian.cache.impl.ReferenceKey;
import com.atlassian.cache.impl.StrongSupplier;
import com.atlassian.cache.impl.WeakSupplier;
import com.atlassian.cache.impl.jmx.MBeanRegistrar;
import com.atlassian.cache.impl.metrics.InstrumentedCache;
import com.atlassian.cache.impl.metrics.InstrumentedCachedReference;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.management.ManagementService;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.management.MBeanServer;
import java.util.function.Supplier;

/**
 * Maintains a mapping of name -&gt; Cache and provides factory methods for creating ad getting caches.
 *
 * @since 2.0
 */
@ParametersAreNonnullByDefault
public class EhCacheManager extends AbstractCacheManager implements MBeanRegistrar
{
    private final net.sf.ehcache.CacheManager delegate;
    private final @Nullable
    EhCacheReplicatorConfigFactory replicatorConfigFactory;

    private final ValueProcessor valueProcessor;

    private boolean statisticsEnabled = true;

    /**
     * Creates an instance backed by a new instance of {@link net.sf.ehcache.CacheManager} and a {@link com.atlassian.cache.ehcache.replication.rmi.RMICacheReplicatorConfigFactory}.
     *
     * @deprecated Since 5.6.9 Use {@link #EhCacheManager(CacheManager, EhCacheReplicatorConfigFactory, CacheSettingsDefaultsProvider)}
     */
    @Deprecated
    public EhCacheManager()
    {
        this(net.sf.ehcache.CacheManager.create(), null);
    }

    /**
     * Creates an instance backed by the provided instance of {@link net.sf.ehcache.CacheManager} and a {@link com.atlassian.cache.ehcache.replication.rmi.RMICacheReplicatorConfigFactory}.
     *
     * @param delegate an Ehcache's cache manager
     *
     * @deprecated since 2.6.0 Use {@link #EhCacheManager(CacheManager, EhCacheReplicatorConfigFactory, CacheSettingsDefaultsProvider)}
     */
    @Deprecated
    public EhCacheManager(net.sf.ehcache.CacheManager delegate, @Nullable CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider)
    {
        this(delegate, new RMICacheReplicatorConfigFactory(), cacheSettingsDefaultsProvider, null);
    }

    /**
     * Creates an instance backed by the provided instance of {@link net.sf.ehcache.CacheManager}.
     *
     * @param delegate an Ehcache's cache manager
     */
    public EhCacheManager(net.sf.ehcache.CacheManager delegate,
                          @Nullable EhCacheReplicatorConfigFactory replicatorConfigFactory,
                          @Nullable CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider)
    {
        this (delegate, replicatorConfigFactory, cacheSettingsDefaultsProvider, null);
    }

    /**
     * Creates an instance backed by the provided instance of {@link net.sf.ehcache.CacheManager}.
     *
     * @param delegate an Ehcache's cache manager
     */
    public EhCacheManager(net.sf.ehcache.CacheManager delegate,
                          @Nullable EhCacheReplicatorConfigFactory replicatorConfigFactory,
                          @Nullable CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider,
                          @Nullable ValueProcessor valueProcessor)
    {
        this(delegate, replicatorConfigFactory, cacheSettingsDefaultsProvider, valueProcessor, new CacheManagerMetricEmitter());
    }

    /**
     * Creates an instance backed by the provided instance of {@link net.sf.ehcache.CacheManager}.
     *
     * @param delegate an Ehcache's cache manager
     */
    @VisibleForTesting
    EhCacheManager(net.sf.ehcache.CacheManager delegate,
                          @Nullable EhCacheReplicatorConfigFactory replicatorConfigFactory,
                          @Nullable CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider,
                          @Nullable ValueProcessor valueProcessor,
                          @Nonnull CacheManagerMetricEmitter metricEmitter)
    {
        super(cacheSettingsDefaultsProvider, metricEmitter);
        this.delegate = delegate;
        this.replicatorConfigFactory = replicatorConfigFactory;
        this.valueProcessor = valueProcessor != null ? valueProcessor : new NoopValueProcessor();
    }

    net.sf.ehcache.CacheManager getEh()
    {
        return delegate;
    }

    public boolean isStatisticsEnabled()
    {
        return statisticsEnabled;
    }

    public void setStatisticsEnabled(final boolean statisticsEnabled)
    {
        this.statisticsEnabled = statisticsEnabled;
    }

    @Nonnull
    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull final String name,
                                                     @Nonnull final com.atlassian.cache.Supplier<V> supplier,
                                                     @Nonnull final CacheSettings settings)
    {
        // Force the cache settings to be flushable and a maximum size of one.
        final CacheSettings overridenSettings = settings.override(
                new CacheSettingsBuilder().flushable().maxEntries(1).build());

        return cacheCreationLocks.apply(name).withLock((Supplier<CachedReference<V>>) () -> {
            final Ehcache spCache = getLoadingCache(name, overridenSettings, new ValueProcessorAtlassianCacheLoaderDecorator(new SupplierAdapter<>(supplier), valueProcessor));
            final InstrumentedCachedReference<V> cache = InstrumentedCachedReference.wrap(
                    DelegatingCachedReference.create(spCache, overridenSettings, valueProcessor)
            );
            caches.put(name, new WeakSupplier<>(cache));
            return cache;
        });
    }

    @Override
    protected ManagedCache createSimpleCache(@Nonnull final String name, @Nonnull final CacheSettings settings)
    {
        final Supplier<ManagedCache> cacheSupplier = caches.get(name);
        if (cacheSupplier != null)
        {
            ManagedCache cache = cacheSupplier.get();
            if (cache != null)
            {
                return cache;
            }
        }
        return createManagedCacheInternal(name, settings);
    }

    private ManagedCache createManagedCacheInternal(@Nonnull final String name, @Nonnull final CacheSettings settings) {
        return cacheCreationLocks.apply(name).withLock((Supplier<ManagedCache>) () -> {
            final ManagedCache result = caches.get(name) == null ? null : caches.get(name).get();
            if (result == null)
            {
                final Ehcache simpleCache = createCache(name, settings, false);
                final InstrumentedCache<Object, Object> cache =
                        InstrumentedCache.wrap(DelegatingCache.create(simpleCache, settings, valueProcessor));
                caches.put(name,  new StrongSupplier<>(cache));
                return cache;
            }
            return result;
        });
    }

    protected <K, V> ManagedCache createComputingCache(@Nonnull final String name, @Nonnull final CacheSettings settings, final CacheLoader<K, V> loader)
    {
        return cacheCreationLocks.apply(name).withLock((Supplier<ManagedCache>) () -> {
            Ehcache spCache = getLoadingCache(name, settings, new ValueProcessorAtlassianCacheLoaderDecorator(loader, valueProcessor));
            final InstrumentedCache<Object, Object> cache =
                    InstrumentedCache.wrap(DelegatingCache.create(spCache, settings, valueProcessor));
            caches.put(name,  new WeakSupplier<>(cache));
            return cache;
        });
    }

    private <K, V> Ehcache getLoadingCache(final @Nonnull String name, final @Nonnull CacheSettings settings, final CacheLoader<K, V> loader)
    {
        final Ehcache ehcache = getCleanCache(name, settings);
        final SynchronizedLoadingCacheDecorator decorator;
        if (ehcache instanceof SynchronizedLoadingCacheDecorator) {
            decorator = (SynchronizedLoadingCacheDecorator) ehcache;
        } else {
            decorator = new SynchronizedLoadingCacheDecorator(ehcache);
            delegate.replaceCacheWithDecoratedCache(ehcache, decorator);
        }
        return new LoadingCache<>(decorator, loader);

    }

    private Ehcache getCleanCache(final String name, final CacheSettings settings)
    {
        // Particularly for plugins that are reloaded the cache may already exist from earlier times.
        // Because actually removing the old cache and recreating it breaks RMI replication we get the old one,
        // but to avoid any class loading issues with the old plugin class loader which would be unavailable
        // we remove all entries from the local version of the cache

        Ehcache ehCache = delegate.getEhcache(name);
        if (ehCache != null)
        {
            // Remove all entries but do not notify replication listeners.
            ehCache.removeAll(true);
        }
        else
        {
            ehCache = createCache(name, settings, true);
        }
        return ehCache;
    }

    private Ehcache createCache(String name, CacheSettings settings, boolean selfLoading)
    {
        return new EhCacheHelper(replicatorConfigFactory).getEhcache(name, delegate, settings, selfLoading, statisticsEnabled);
    }

    @Override
    public void shutdown()
    {

        delegate.shutdown();
    }

    @Override
    public void registerMBeans(@Nullable final MBeanServer mBeanServer)
    {
        if (mBeanServer != null)
        {
            ManagementService.registerMBeans(delegate, mBeanServer, true, true, true, true);
        }
    }

    @Override
    public void unregisterMBeans(@Nullable final MBeanServer mBeanServer)
    {
        if (mBeanServer != null)
        {
            final ManagementService managementService = new ManagementService(delegate, mBeanServer, true, true, true, true);
            managementService.dispose();
        }
    }

    static class SupplierAdapter<V> implements CacheLoader<ReferenceKey, V>
    {
        private final com.atlassian.cache.Supplier<V> supplier;

        SupplierAdapter(com.atlassian.cache.Supplier<V> supplier)
        {
            this.supplier = supplier;
        }

        @Nonnull
        @Override
        public V load(@Nonnull ReferenceKey key)
        {
            return supplier.get();
        }
    }
}
