package com.atlassian.cache.ehcache;

import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.ManagedCache;

import net.sf.ehcache.Ehcache;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;

@ParametersAreNonnullByDefault
abstract class ManagedCacheSupport implements ManagedCache
{
    private final @Nonnull Ehcache delegate;
    private final @Nonnull CacheSettings settings;

    public ManagedCacheSupport(Ehcache delegate, CacheSettings settings)
    {
        this.delegate = checkNotNull(delegate, "Ehcache delegate cannot be null");
        this.settings = checkNotNull(settings, "CacheSettings cannot be null");
    }

    @Nonnull
    @Override
    public final String getName()
    {
        return delegate.getName();
    }

    @Override
    public final boolean isFlushable()
    {
        return settings.getFlushable(true);
    }

    @Override
    public final boolean isLocal()
    {
        return settings.getLocal(false);
    }

    @Override
    public final boolean isReplicateAsynchronously()
    {
        return settings.getReplicateAsynchronously(false);
    }

    @Override
    public final boolean isReplicateViaCopy()
    {
        return settings.getReplicateViaCopy(false);
    }

    @Override
    @Nonnull
    public final Integer currentMaxEntries()
    {
        return (int) delegate.getCacheConfiguration().getMaxEntriesLocalHeap();
    }

    @Override
    public final boolean updateMaxEntries(int newValue)
    {
        delegate.getCacheConfiguration().setMaxEntriesLocalHeap(newValue);
        return true;
    }

    @Override
    @Nonnull
    public final Long currentExpireAfterAccessMillis()
    {
        return MILLISECONDS.convert(
            delegate.getCacheConfiguration().getTimeToIdleSeconds(), SECONDS);
    }

    @Override
    public final boolean updateExpireAfterAccess(long expireAfter, @Nonnull TimeUnit timeUnit)
    {
        delegate.getCacheConfiguration().setTimeToIdleSeconds(timeUnit.toSeconds(expireAfter));
        return true;
    }

    @Override
    @Nonnull
    public final Long currentExpireAfterWriteMillis()
    {
        return MILLISECONDS.convert(
            delegate.getCacheConfiguration().getTimeToLiveSeconds(), SECONDS);
    }

    @Override
    public final boolean updateExpireAfterWrite(long expireAfter, @Nonnull TimeUnit timeUnit)
    {
        delegate.getCacheConfiguration().setTimeToLiveSeconds(timeUnit.toSeconds(expireAfter));
        return true;
    }

    @Override
    public boolean isStatisticsEnabled()
    {
        return settings.getStatisticsEnabled(true);
    }

    @Override
    public void setStatistics(boolean enabled)
    {
        throw new UnsupportedOperationException("setStatistics() not implemented");
    }
}