package com.atlassian.cache.compat.impl;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.atlassian.cache.compat.Cache;
import com.atlassian.cache.compat.CacheLoader;
import com.atlassian.cache.compat.CacheFactory;
import com.atlassian.cache.compat.CacheSettings;
import com.atlassian.cache.compat.CacheSettingsBuilder;
import com.atlassian.cache.compat.CachedReference;
import com.atlassian.cache.compat.Supplier;
import java.util.function.Function;
import io.atlassian.util.concurrent.ManagedLock;
import io.atlassian.util.concurrent.ManagedLocks;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A partial implementation of {@link CacheFactory}.
 *
 * @since v1.0
 */
public abstract class AbstractCacheFactory implements CacheFactory
{

    /**
     * Map of all the caches.
     * Before creating a cache to put into the map, acquire a lock using {@link #cacheCreationLocks}
     * to ensure that a cache is created only once.
     */
    protected final ConcurrentMap<String, Supplier<Cache>> caches = new ConcurrentHashMap<String, Supplier<Cache>>();
    /** Used to synchronize the creation of caches. */
    protected final Function<String, ManagedLock> cacheCreationLocks = ManagedLocks.weakManagedLockFactory();



    @Override
    public <K,V> Cache<K,V> getCache(@Nonnull final String name)
    {
        return getCache(name, null);
    }

    @Override
    public <K,V> Cache<K,V> getCache(Class<?> owningClass, String name)
    {
        return getCache(cacheName(owningClass, name));
    }

    @Override
    public <K,V> Cache<K,V> getCache(final String name, final Class<K> keyType, final Class<V> valueType)
    {
        return getCache(name);
    }

    @Override
    public <K, V> Cache<K, V> getCache(String name, CacheLoader<K, V> loader)
    {
        return getCache(name, loader, new CacheSettingsBuilder().build());
    }

    @Override
    public <V> CachedReference<V> getCachedReference(String name, Supplier<V> supplier)
    {
        return getCachedReference(name, supplier, new CacheSettingsBuilder().build());
    }

    @Override
    public <V> CachedReference<V> getCachedReference(Class<?> owningClass, String name, Supplier<V> supplier)
    {
        return getCachedReference(owningClass, name, supplier, new CacheSettingsBuilder().build());
    }

    @Override
    public <V> CachedReference<V> getCachedReference(Class<?> owningClass,
                                                     String name,
                                                     Supplier<V> supplier,
                                                     CacheSettings settings)
    {
        return getCachedReference(cacheName(owningClass, name), supplier, settings);
    }

    private static String cacheName(Class<?> owningClass, String name)
    {
        checkNotNull(name, "name cannot be null");
        return owningClass.getName() + "." + name;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <K, V> Cache<K, V> getCache(String name, CacheLoader<K, V> loader, CacheSettings settings)
    {
        if (null == loader)
        {
            return createSimpleCache(name, settings);
        }
        else
        {
            return createComputingCache(name, settings, loader);
        }
    }

    /**
     * Creates a cache that upon a miss is able to populate itself using the loader.
     *
     * @return a non-null cache
     */
    protected abstract <K,V> Cache<K,V> createComputingCache(String name, CacheSettings settings, CacheLoader<K, V> loader);

    /**
     * Creates a cache with no loader, i.e. one populated via explicit puts.
     *
     * @param name the name to give the cache (required)
     * @return a non-null cache
     */
    protected abstract <K,V> Cache<K,V> createSimpleCache(String name, CacheSettings settings);
}
