package com.atlassian.cache.impl.metrics;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.cache.CacheStatisticsKey;
import com.atlassian.cache.CachedReference;
import com.atlassian.cache.CachedReferenceListener;
import com.atlassian.cache.ManagedCache;
import com.atlassian.instrumentation.caches.CacheCollector;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;
import java.util.SortedMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;
/**
 * This class wraps a CachedReference for the purpose of emitting JMX metrics
 * @since 5.5.0
 */
public class InstrumentedCachedReference<V> implements CachedReference<V>, ManagedCache
{

    private final CachedReference<V> delegate;
    private final MetricEmitter metricEmitter;

    @VisibleForTesting
    InstrumentedCachedReference(CachedReference<V> delegate,
                                MetricEmitter metricEmitter)
    {
        this.delegate = delegate;
        this.metricEmitter = metricEmitter;
    }

    public static <V> InstrumentedCachedReference<V> wrap(@Nonnull CachedReference<V> delegate)
    {
        requireNonNull(delegate, "delegate");
        if (!(delegate instanceof ManagedCache))
        {
            throw new MissingInterfacesException(delegate, ManagedCache.class);
        }

        MetricEmitter metricEmitter = MetricEmitter.create(delegate.getClass().getName());
        return new InstrumentedCachedReference<>(delegate, metricEmitter);
    }

    @VisibleForTesting
    MetricEmitter getMetricEmitter()
    {
        return metricEmitter;
    }

    @Override
    @Nonnull
    public V get()
    {
        return delegate.get();
    }

    @Override
    public void reset()
    {
        metricEmitter.emitCachedReferenceReset();
        delegate.reset();
    }

    @Override
    public boolean isPresent()
    {
        return delegate.isPresent();
    }

    @Override
    @Nonnull
    public Optional<V> getIfPresent()
    {
        return delegate.getIfPresent();
    }

    @Override
    public void addListener(@Nonnull CachedReferenceListener<V> listener, boolean includeValues)
    {
        delegate.addListener(listener, includeValues);
    }

    @Override
    public void removeListener(@Nonnull CachedReferenceListener<V> listener)
    {
        delegate.removeListener(listener);
    }

    @Override
    public void clear()
    {
        ((ManagedCache) delegate).clear();
    }

    @Override
    @Nonnull
    public String getName()
    {
        return ((ManagedCache) delegate).getName();
    }

    @Override
    public boolean isFlushable()
    {
        return ((ManagedCache) delegate).isFlushable();
    }

    @Override
    @Nullable
    public Integer currentMaxEntries()
    {
        return ((ManagedCache) delegate).currentMaxEntries();
    }

    @Override
    public boolean updateMaxEntries(int newValue)
    {
        return ((ManagedCache) delegate).updateMaxEntries(newValue);
    }

    @Override
    @Nullable
    public Long currentExpireAfterAccessMillis()
    {
        return ((ManagedCache) delegate).currentExpireAfterAccessMillis();
    }

    @Override
    public boolean updateExpireAfterAccess(long expireAfter, @Nonnull TimeUnit timeUnit)
    {
        return ((ManagedCache) delegate).updateExpireAfterAccess(expireAfter, timeUnit);
    }

    @Override
    @Nullable
    public Long currentExpireAfterWriteMillis()
    {
        return ((ManagedCache) delegate).currentExpireAfterWriteMillis();
    }

    @Override
    public boolean updateExpireAfterWrite(long expireAfter, @Nonnull TimeUnit timeUnit)
    {
        return ((ManagedCache) delegate).updateExpireAfterWrite(expireAfter, timeUnit);
    }

    @Override
    public boolean isLocal()
    {
        return ((ManagedCache) delegate).isLocal();
    }

    @Override
    public boolean isReplicateAsynchronously()
    {
        return ((ManagedCache) delegate).isReplicateAsynchronously();
    }

    @Override
    public boolean isReplicateViaCopy()
    {
        return ((ManagedCache) delegate).isReplicateViaCopy();
    }

    @Override
    public boolean isStatisticsEnabled()
    {
        return ((ManagedCache) delegate).isStatisticsEnabled();
    }

    @Override
    @Deprecated
    public void setStatistics(boolean enabled) {
        ((ManagedCache) delegate).setStatistics(enabled);
    }

    @Override
    @Nonnull
    public SortedMap<CacheStatisticsKey, Supplier<Long>> getStatistics()
    {
        return ((ManagedCache) delegate).getStatistics();
    }

    @Override
    @Nullable
    public CacheCollector getCacheCollector()
    {
        return ((ManagedCache) delegate).getCacheCollector();
    }
}
