package com.atlassian.cache.impl.metrics;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheEntryListener;
import com.atlassian.cache.CacheStatisticsKey;
import com.atlassian.cache.ManagedCache;
import com.atlassian.cache.Supplier;
import com.atlassian.instrumentation.caches.CacheCollector;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import static java.util.Objects.requireNonNull;
/**
 * This class wraps a Cache for the purpose of emitting JMX metrics
 * @since 5.5.0
 */
public class InstrumentedCache<K,V> implements Cache<K, V>, ManagedCache
{

    private final Cache<K,V> delegate;
    private final MetricEmitter metricEmitter;

    @VisibleForTesting
    InstrumentedCache(Cache<K, V> delegate, MetricEmitter metricEmitter)
    {
        this.delegate = delegate;
        this.metricEmitter = metricEmitter;
    }

    public static <K,V> InstrumentedCache<K,V> wrap(@Nonnull Cache<K,V> delegate)
    {
        requireNonNull(delegate, "delegate");
        if (!(delegate instanceof ManagedCache))
        {
            throw new MissingInterfacesException(delegate, ManagedCache.class);
        }

        MetricEmitter metricEmitter = MetricEmitter.create(delegate.getClass().getName());
        return new InstrumentedCache<>(delegate, metricEmitter);
    }

    @Override
    @Nonnull
    public String getName()
    {
        return delegate.getName();
    }

    public boolean containsKey(@Nonnull K key)
    {
        return delegate.containsKey(key);
    }

    @Override
    @Nonnull
    public Collection<K> getKeys()
    {
        return delegate.getKeys();
    }

    @Nullable
    public V get(@Nonnull K key)
    {
        return delegate.get(key);
    }

    @Nonnull
    public V get(@Nonnull K key, @Nonnull Supplier<? extends V> valueSupplier)
    {
        return delegate.get(key, valueSupplier);
    }

    @Nonnull
    public Map<K, V> getBulk(@Nonnull final Set<K> keys, @Nonnull final Function<Set<K>, Map<K, V>> valuesSupplier)
    {
        return delegate.getBulk(keys, valuesSupplier);
    }

    public void put(@Nonnull K key, @Nonnull V value)
    {
        delegate.put(key, value);
    }

    @Nullable
    public V putIfAbsent(@Nonnull K key, @Nonnull V value)
    {
        return delegate.putIfAbsent(key, value);
    }

    public void remove(@Nonnull K key)
    {
        delegate.remove(key);
    }

    public boolean remove(@Nonnull K key, @Nonnull V value)
    {
        return delegate.remove(key, value);
    }

    @Override
    public void removeAll()
    {
        metricEmitter.emitCacheRemoveAll();
        delegate.removeAll();
    }

    public boolean replace(@Nonnull K key, @Nonnull V oldValue,
                           @Nonnull V newValue)
    {
        return delegate.replace(key, oldValue, newValue);
    }

    public void addListener(@Nonnull CacheEntryListener<K,V> listener,
                            boolean includeValues)
    {
        delegate.addListener(listener, includeValues);
    }

    public void removeListener(@Nonnull CacheEntryListener<K,V> listener)
    {
        delegate.removeListener(listener);
    }

    @Override
    public void clear()
    {
        ((ManagedCache) delegate).clear();
    }

    @Override
    public boolean isFlushable()
    {
        return ((ManagedCache) delegate).isFlushable();
    }

    @Override
    @Nullable
    public Integer currentMaxEntries()
    {
        return ((ManagedCache) delegate).currentMaxEntries();
    }

    @Override
    public boolean updateMaxEntries(int newValue)
    {
        return ((ManagedCache) delegate).updateMaxEntries(newValue);
    }

    @Override
    @Nullable
    public Long currentExpireAfterAccessMillis()
    {
        return ((ManagedCache) delegate).currentExpireAfterAccessMillis();
    }

    @Override
    public boolean updateExpireAfterAccess(long expireAfter, @Nonnull
            TimeUnit timeUnit)
    {
        return ((ManagedCache) delegate).updateExpireAfterAccess(expireAfter, timeUnit);
    }

    @Override
    @Nullable
    public Long currentExpireAfterWriteMillis()
    {
        return ((ManagedCache) delegate).currentExpireAfterWriteMillis();
    }

    @Override
    public boolean updateExpireAfterWrite(long expireAfter, @Nonnull
            TimeUnit timeUnit)
    {
        return ((ManagedCache) delegate).updateExpireAfterWrite(expireAfter, timeUnit);
    }

    @Override
    public boolean isLocal()
    {
        return ((ManagedCache) delegate).isLocal();
    }

    @Override
    public boolean isReplicateAsynchronously()
    {
        return ((ManagedCache) delegate).isReplicateAsynchronously();
    }

    @Override
    public boolean isReplicateViaCopy()
    {
        return ((ManagedCache) delegate).isReplicateViaCopy();
    }

    @Override
    public boolean isStatisticsEnabled()
    {
        return ((ManagedCache) delegate).isStatisticsEnabled();
    }

    @Override
    @Deprecated
    public void setStatistics(boolean enabled) {
        ((ManagedCache) delegate).setStatistics(enabled);
    }

    @Override
    @Nonnull
    public SortedMap<CacheStatisticsKey, java.util.function.Supplier<Long>> getStatistics()
    {
        return ((ManagedCache) delegate).getStatistics();
    }

    @Override
    @Nullable
    public CacheCollector getCacheCollector()
    {
        return ((ManagedCache) delegate).getCacheCollector();
    }

    @VisibleForTesting
    MetricEmitter getMetricEmitter() {
        return metricEmitter;
    }
}
