package com.atlassian.cache.impl;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.AbstractQueuedSynchronizer;

/**
 * A useful synchronization construct for ensuring the correct behaviour of cache implementations.
 * <p>
 * The one-shot latch is initially held by the thread that creates it and must be explicitly
 * released by it.
 * </p>
 *
 * @since v2.4.5
 */
public class OneShotLatch
{
    private final Sync sync = new Sync();

    public boolean isHeldByCurrentThread()
    {
        return sync.isHeldByCurrentThread();
    }

    public void release()
    {
        sync.release(0);
    }

    public void await()
    {
        sync.acquireShared(0);
    }

    public boolean await(long timeout, TimeUnit unit) throws InterruptedException
    {
        return sync.tryAcquireSharedNanos(0, unit.toNanos(timeout));
    }

    @Override
    public String toString()
    {
        return "OneShotLatch[sync=" + sync + ']';
    }



    static class Sync extends AbstractQueuedSynchronizer
    {
        private static final int STATE_UNAVAILABLE = -1;
        private static final int STATE_RELEASED = 1;

        Sync()
        {
            setState(STATE_UNAVAILABLE);
            setExclusiveOwnerThread(Thread.currentThread());
        }

        boolean isHeldByCurrentThread()
        {
            return getExclusiveOwnerThread() == Thread.currentThread();
        }

        @Override
        protected boolean tryRelease(int ignored)
        {
            if (!isHeldByCurrentThread())
            {
                throw new IllegalMonitorStateException(toString());
            }

            setExclusiveOwnerThread(null);
            setState(STATE_RELEASED);
            return true;
        }

        @Override
        protected int tryAcquireShared(int ignored)
        {
            final int state = getState();
            if (state == STATE_UNAVAILABLE && isHeldByCurrentThread())
            {
                throw new IllegalMonitorStateException(toString());
            }
            return state;
        }

        @Override
        public String toString()
        {
            return "Sync[state=" + getState() + "; owner=" + getExclusiveOwnerThread() + ']';
        }
    }
}
