package com.atlassian.cache;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.annotations.PublicApi;

/**
 * An immutable representation of settings for a Cache. The property getters will return
 * <code>null</code> to indicate that no setting is specified. The caller is
 * responsible for handling that <code>null</code> may be returned.
 * @since 2.0
 */
@PublicApi
public interface CacheSettings
{
    /**
     * @return how long to retain entries (in milliseconds) from when they were last accessed, or <code>null</code>.
     */
    @Nullable
    Long getExpireAfterAccess();

    /**
     * Convenience method that returns {@link #getExpireAfterAccess()} if it is not <code>null</code>,
     * otherwise returns the supplied <code>defaultValue</code>.
     */
    long getExpireAfterAccess(long defaultValue);

    /**
     * @return how long to retain entries (in milliseconds) from when they were created, or <code>null</code>.
     */
    @Nullable
    Long getExpireAfterWrite();

    /**
     * Convenience method that returns {@link #getExpireAfterWrite()} if it is not <code>null</code>,
     * otherwise returns the supplied <code>defaultValue</code>.
     */
    long getExpireAfterWrite(long defaultValue);

    /**
     * @return whether this cache can be flushed by the cache manager when desired, or <code>null</code>.
     */
    @Nullable
    Boolean getFlushable();

    /**
     * Convenience method that returns {@link #getFlushable()} if it is not <code>null</code>,
     * otherwise returns the supplied <code>defaultValue</code>.
     */
    boolean getFlushable(boolean defaultValue);

    /**
     * @return whether this cache should be local to the node (jvm) where the cache is created, or <code>null</code>.
     */
    @Nullable
    Boolean getLocal();

    /**
     * Convenience method that returns {@link #getLocal()} if it is not <code>null</code>,
     * otherwise returns the supplied <code>defaultValue</code>.
     */
    boolean getLocal(boolean defaultValue);

    /**
     * @return the maximum number of entries in the cache, or <code>null</code>.
     */
    @Nullable
    Integer getMaxEntries();

    /**
     * Convenience method that returns {@link #getMaxEntries()} if it is not <code>null</code>,
     * otherwise returns the supplied <code>defaultValue</code>.
     */
    int getMaxEntries(int defaultValue);

    /**
     * @return whether this cache should replicate asynchronously, or <code>null</code>.
     */
    @Nullable
    Boolean getReplicateAsynchronously();

    /**
     * Convenience method that returns {@link #getReplicateAsynchronously()} if it is not <code>null</code>,
     * otherwise returns the supplied <code>defaultValue</code>.
     */
    boolean getReplicateAsynchronously(boolean defaultValue);

    /**
     * @return whether this cache should replicate put and update operations by copying the relevant
     * key and value across the wire, or <code>null</code>.
     */
    @Nullable
    Boolean getReplicateViaCopy();

    /**
     * Convenience method that returns {@link #getReplicateViaCopy()} if it is not <code>null</code>,
     * otherwise returns the supplied <code>defaultValue</code>.
     */
    boolean getReplicateViaCopy(boolean defaultValue);

    /**
     * @return whether this cache should record statistics or not, or <code>null</code>.
     */
    @Nullable
    Boolean getStatisticsEnabled();

    /**
     * Convenience method that returns {@link #getStatisticsEnabled()} ()} if it is not <code>null</code>,
     * otherwise returns the supplied <code>defaultValue</code>.
     */
    boolean getStatisticsEnabled(boolean defaultValue);

    /**
     * Returns a new {@link CacheSettings} instance where the current settings
     * are overridden with settings specified in <code>overrides</code>. Only properties
     * in <code>overrides</code> that are not <code>null</code> will be applied.
     * @param overrides contains the settings to override
     * @return a new {@link CacheSettings} instance with the <code>overrides</code> settings applied.
     */
    @Nonnull
    CacheSettings override(@Nonnull CacheSettings overrides);
}
