package com.atlassian.cache;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Interface that defines the common cache operations.
 * <p>
 * Note: {@code null} keys and values are NOT supported. Although using {@code null} values may work for some
 * implementations, it won't work for all implementations and the behaviour may change over time.
 *
 * @param <K> the key type
 * @param <V> the value type
 */
@PublicApi
public interface Cache<K, V> extends ReadThroughCache<K, V>
{
    /**
     * Put an object into the cache. If the specified key already exists within the cache, it will be replaced by the
     * new object.
     * <p>
     * <b>NOTE:</b> This method should not be used on {@link CacheSettingsBuilder#replicateViaInvalidation() hybrid}
     * caches that maintain consistency only by invalidating the key on other cluster nodes. Instead, such hybrid
     * caches should use {@link CacheLoader} semantics and supply a {@link CacheLoader} when getting the {@link Cache}.
     *
     * @param key the key uniquely identifying the object to be added into the cache
     * @param value the non-null value to be cached
     */
    void put(@Nonnull K key, @Nonnull V value);

    /**
     * Atomically associates the specified key with the given value if it is not already associated with a value.
     * <p>
     * <b>NOTE:</b> This method should not be used on {@link CacheSettingsBuilder#replicateViaInvalidation() hybrid}
     * caches that maintain consistency only by invalidating the key on other cluster nodes. Instead, such hybrid
     * caches should use {@link CacheLoader} semantics and supply a {@link CacheLoader} when getting the {@link Cache}.
     *
     * @param key the key with which the specified value is associated
     * @param value the non-null value to be cached
     * @return the previous value associated with the specified key, or {@code null} if there was no mapping for the key
     */
    @Nullable
    V putIfAbsent(@Nonnull K key, @Nonnull V value);

    /**
     * Atomically replaces the entry for a key only if currently mapped to a given value.
     * <p>
     * <b>NOTE:</b> Some cache backends (e.g. Ehcache) require this operation to be performed by CAS and will throw an
     * exception when the cache is replicated
     *
     * @param key the key with which the specified value is associated
     * @param oldValue the value expected to be associated with the specified key
     * @param newValue the value to be associated with the specified key
     * @return {@code true} if the value was replaced, {@code false} otherwise
     */
    boolean replace(@Nonnull K key, @Nonnull V oldValue, @Nonnull V newValue);

    /**
     * Adds a {@link CacheEntryListener}
     * @param listener the listener
     * @param includeValues if the events sent to this listener will include old/new value. This is can be used
     * in cases when the cost of finding these values is big (network sync) but the listener is not interested in
     * the concrete values for events its getting. The support for this parameter is optional and implementation dependant
     *
     * @throws UnsupportedOperationException if not supported
     * @since 2.4
     */
    void addListener(@Nonnull CacheEntryListener<K, V> listener, boolean includeValues);

    /**
     * Removes a {@link CacheEntryListener}
     * @param listener the listener
     *
     * @throws UnsupportedOperationException if not supported
     * @since 2.4
     */
    void removeListener(@Nonnull CacheEntryListener<K, V> listener);
}
