package com.atlassian.cache;

import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.annotations.PublicApi;

/**
 * Interface that defines the common cache operations.
 * <p>
 * Note: {@code null} keys and values are NOT supported. Although using {@code null} values may work for some 
 * implementations, it won't work for all implementations and the behaviour may change over time.
 *
 * @param <K> the key type
 * @param <V> the value type
 */
@PublicApi
public interface Cache<K, V>
{
    /**
     * The name of the cache, uniquely identifies this cache.
     *
     * @return the name of the cache
     */
    @Nonnull
    String getName();

    /**
     * Returns whether an entry exists in the cache under the specified key.
     *
     * <p>Note that:</p>
     *
     * <ul>
     *     <li>If the cache was created with a {@link com.atlassian.cache.CacheLoader},
     *     it will not be called. Obviously, any call to {@link #get(Object)}
     *     will call the corresponding {@link com.atlassian.cache.CacheLoader} (if required).</li>
     *     <li>If the cache was created with {@link CacheSettings#getReplicateViaCopy()}
     *     set to <tt>false</tt> and {@link CacheSettings#getLocal()} set to <tt>false</tt>,
     *     then only the local copy of the cache is checked. A local cache on
     *     another node may contain an entry under the specified key.</li>
     * </ul>
     * @param key the key for the entry to check for containment
     * @return true iff the cache already contains an entry under the specified key
     * @since 2.2.0
     */
    boolean containsKey(@Nonnull K key);

    /**
     * Gets the keys of all objects currently stored in the cache. This will return the keys in a new collection.
     * @return a collection of {@link Object}s keys
     */
    @Nonnull
    Collection<K> getKeys();

    /**
     * Retrieve an object from this cache. Note that a copy of the cached object may be returned. Any changes that are
     * made to the returned object may not be reflected in the cache. Cached values should be considered effectively
     * immutable.
     *
     * @param key the key uniquely identifying the object to be retrieved
     * @return the object from the cache, or {@code null}  if the object is not found
     */
    @Nullable
    V get(@Nonnull K key);

    /**
     * Retrieve an object from this cache. Note that a copy of the cached object may be returned. Any changes that are
     * made to the returned object may not be reflected in the cache. Cached values should be considered effectively
     * immutable.
     * <p>
     * If no value is present in the cache, the {@code valueSupplier} will be used to populate the entry and be counted
     * as a cache miss.
     *
     * @param key the key uniquely identifying the object to be retrieved
     * @param valueSupplier the supplier to call if no value is stored in the cache. the value supplied by the supplier
     * cannot be {@code null}
     * @return the object from the cache, or the newly created value from the supplier
     * @since 2.5
     */
    @Nonnull
    V get(@Nonnull K key, @Nonnull Supplier<? extends V> valueSupplier);

    /**
     * Put an object into the cache. If the specified key already exists within the cache, it will be replaced by the
     * new object.
     * <p>
     * <b>NOTE:</b> This method should not be used on {@link CacheSettingsBuilder#replicateViaInvalidation() hybrid}
     * caches that maintain consistency only by invalidating the key on other cluster nodes. Instead, such hybrid
     * caches should use {@link CacheLoader} semantics and supply a {@link CacheLoader} when getting the {@link Cache}.
     *
     * @param key the key uniquely identifying the object to be added into the cache
     * @param value the non-null value to be cached
     */
    void put(@Nonnull K key, @Nonnull V value);

    /**
     * Atomically associates the specified key with the given value if it is not already associated with a value.
     * <p>
     * <b>NOTE:</b> This method should not be used on {@link CacheSettingsBuilder#replicateViaInvalidation() hybrid}
     * caches that maintain consistency only by invalidating the key on other cluster nodes. Instead, such hybrid
     * caches should use {@link CacheLoader} semantics and supply a {@link CacheLoader} when getting the {@link Cache}.
     *
     * @param key the key with which the specified value is associated
     * @param value the non-null value to be cached
     * @return the previous value associated with the specified key, or {@code null} if there was no mapping for the key
     */
    @Nullable
    V putIfAbsent(@Nonnull K key, @Nonnull V value);

    /**
     * Remove the object identified by the key from the cache. If no object can be found associated with this key then
     * no action is taken.
     *
     * @param key the key that uniquely identifies the object to be removed
     */
    void remove(@Nonnull K key);

    /**
     * Atomically removes the entry for a key only if currently mapped to a given value.
     *
     * @param key the key with which the specified value is associated
     * @param value the value expected to be associated with the specified key
     * @return {@code true} if the value was removed, {@code false}  otherwise
     */
    boolean remove(@Nonnull K key, @Nonnull V value);

    /**
     * Remove all of the objects from this cache.
     */
    void removeAll();

    /**
     * Atomically replaces the entry for a key only if currently mapped to a given value.
     *
     * @param key the key with which the specified value is associated
     * @param oldValue the value expected to be associated with the specified key
     * @param newValue the value to be associated with the specified key
     * @return {@code true} if the value was replaced, {@code false} otherwise
     */
    boolean replace(@Nonnull K key, @Nonnull V oldValue, @Nonnull V newValue);

    /**
     * Adds a {@link CacheEntryListener}
     * @param listener the listener
     * @param includeValues if the events sent to this listener will include old/new value. This is can be used
     * in cases when the cost of finding these values is big (network sync) but the listener is not interested in
     * the concrete values for events its getting. The support for this parameter is optional and implementation dependant
     *
     * @throws UnsupportedOperationException if not supported
     * @since 2.4
     * @deprecated since 2.9 may not be supported in all implementations and will be removed after June 2016
     */
    @Deprecated
    void addListener(@Nonnull CacheEntryListener<K, V> listener, boolean includeValues);

    /**
     * Removes a {@link CacheEntryListener}
     * @param listener the listener
     *
     * @throws UnsupportedOperationException if not supported
     * @since 2.4
     * @deprecated since 2.9 may not be supported in all implementations and will be removed after June 2016
     */
    @Deprecated
    void removeListener(@Nonnull CacheEntryListener<K, V> listener);
}
