package com.atlassian.cache;

import java.util.Comparator;

import javax.annotation.Nonnull;

import static com.atlassian.cache.StatisticsType.COUNTER;
import static com.atlassian.cache.StatisticsType.GAUGE;
import static com.atlassian.util.concurrent.Assertions.notNull;

/**
 * An immutable representation of a statistics key for a Cache.
 *
 * This class is used by {@link ManagedCache#getStatistics()} to dynamically distinguish different cache
 * statistics available for a {@link com.atlassian.cache.Cache}.
 *
 * @since v2.4.0
 */
public enum CacheStatisticsKey
{
    /**
     * Returns the number of cached objects kept referenced by the {@link Cache} (depends on the cache implementation)
     */
    SIZE("size", GAUGE),

    /**
     * Returns the number of bytes used by the {@link Cache}.
     * <p>
     * Note: This is likely to be a very expensive statistic in some implementations, so consumers may want
     * to discard it when it hasn't been explicitly requested.  Even where available, the reported heap size
     * is almost certain to be only an approximation of the cache's total memory use.
     * </p>
     */
    HEAP_SIZE("heapSize", GAUGE),

    /**
     * Returns the number of times {@link Cache} lookup methods have returned a cached value.
     */
    HIT_COUNT("hitCount", COUNTER),

    /**
     * Returns the number of times {@link Cache} put methods have added a cached value.
     */
    PUT_COUNT("putCount", COUNTER),

    /**
     * Returns the number of times {@link Cache} remove methods have removed a cached value.
     */
    REMOVE_COUNT("removeCount", COUNTER),

    /**
     * Returns the number of times {@link Cache} lookup methods have returned an uncached (newly
     * loaded) value, or null. Multiple concurrent calls to {@link Cache} lookup methods on an absent
     * value can result in multiple misses, all returning the results of a single cache load
     * operation.
     */
    MISS_COUNT("missCount", COUNTER),

    /**
     * Returns the total number of times that Cache lookup methods attempted to load new values.
     */
    LOAD_COUNT("loadCount", COUNTER),

    /**
     * Returns the number of times Cache lookup methods have successfully loaded a new value.
     */
    LOAD_SUCCESS_COUNT("loadSuccessCount", COUNTER),

    /**
     * Returns the number of times Cache lookup methods threw an exception while loading a new value.
     */
    LOAD_EXCEPTION_COUNT("loadExceptionCount", COUNTER),

    /**
     * Returns the number of nanoseconds spent computing values for cache misses. This
     * is only relevant for caches using a CacheLoader.
     */
    TOTAL_MISS_TIME("totalMissTime", COUNTER),

    /**
     * Returns the total number of nanoseconds the cache has spent loading new values.
     */
    TOTAL_LOAD_TIME("totalLoadTime", COUNTER),

    /**
     * Returns the number of times an entry has been evicted.
     */
    EVICTION_COUNT("evictionCount", COUNTER),

    /**
     * Returns the number of times Cache lookup methods have returned either a cached or uncached value.
     */
    REQUEST_COUNT("requestCount", COUNTER);



    /** Sorts cache statistics keys by {@link #getLabel() label}. */
    public static Comparator<CacheStatisticsKey> SORT_BY_LABEL = new Comparator<CacheStatisticsKey>()
    {
        @Override
        public int compare(CacheStatisticsKey key1, CacheStatisticsKey key2)
        {
            return key1.getLabel().compareTo(key2.getLabel());
        }
    };

    private final String label;
    private final StatisticsType type;

    private CacheStatisticsKey(String label, StatisticsType type)
    {
        this.label = notNull("label", label);
        this.type = notNull("type", type);
    }

    /**
     * Returns the label for this cache statistics key.
     * This is intended to be the same as the {@link #name()}, except written in {@code camelCase}.
     * @return the label for this cache statistics key.
     */
    @Nonnull
    public String getLabel()
    {
        return label;
    }

    /**
     * Returns the type of statistic that this is.
     * Different types of statistics might render differently in a user interface.
     * @return the type of statistic that this is.
     */
    @Nonnull
    public StatisticsType getType()
    {
        return type;
    }
}
