package com.atlassian.cache;

import java.util.SortedMap;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.annotations.Internal;
import com.atlassian.instrumentation.caches.CacheCollector;
import com.atlassian.util.concurrent.Supplier;

/**
 * Interface that managed cache things need to implement
 *
 * @since 2.0
 */
@Internal
@SuppressWarnings("UnusedDeclaration")
public interface ManagedCache
{
    /**
     * Clear the cache.
     */
    void clear();

    /**
     * The name of the cache, uniquely identifies this cache.
     *
     * @return the name of the cache.
     */
    @Nonnull
    String getName();

    /**
     * Returns true if this cache can be safely flushed any time.
     * <p>
     * This method is used by {@link CacheManager#flushCaches()} to only flush caches that can be safely flushed. {@link
     * #clear()} ignores this value.
     *
     * @return true if this cache can be flushed safely.
     */
    boolean isFlushable();

    /**
     * @return the current hint regarding the maximum number of entries that should be cached at any time. <tt>null</tt>
     * indicates no current hint
     */
    @Nullable
    Integer currentMaxEntries();

    /**
     * Attempts to update the hint regarding the maximum number of entries that should be cached at any time.
     *
     * @param newValue the new hint value
     * @return <tt>true</tt> iff the hint has been updated to the new value. If a cache does not support changes at
     * run-time, it must return <tt>false</tt>
     */
    boolean updateMaxEntries(int newValue);

    /**
     * @return the current hint regarding how long entries should be held in the cache, after last access, measured in
     * milliseconds. <tt>null</tt> indicates no current hint
     */
    @Nullable
    Long currentExpireAfterAccessMillis();

    /**
     * Attempts to update the hint regarding how long entries that should be held in the cache, after last access.
     *
     * @param expireAfter Time to retain entries for since their last access.
     * @param timeUnit The {@link TimeUnit} for the time
     * @return <tt>true</tt> iff the hint has been updated to the new value. If a cache does not support changes at
     * run-time, it must return <tt>false</tt>
     */
    boolean updateExpireAfterAccess(long expireAfter, @Nonnull TimeUnit timeUnit);

    /**
     * @return the current hint regarding how long entries should be held in the cache, after last write, measured in
     * milliseconds. <tt>null</tt> indicates no current hint
     */
    @Nullable
    Long currentExpireAfterWriteMillis();

    /**
     * Attempts to update the hint regarding how long entries that should be held in the cache, after last write.
     *
     * @param expireAfter Time to retain entries for since their last write.
     * @param timeUnit The {@link TimeUnit} for the time
     * @return <tt>true</tt> iff the hint has been updated to the new value. If a cache does not support changes at
     * run-time, it must return <tt>false</tt>
     */
    boolean updateExpireAfterWrite(long expireAfter, @Nonnull TimeUnit timeUnit);

    /**
     * @return true if the cache is local, and hence is not replicated in a clustered environment
     */
    boolean isLocal();

    /**
     * @return true if this cache should be replicated asynchronously in a clustered environment.
     */
    boolean isReplicateAsynchronously();

    /**
     * @return true if this cache should, in a clustered environment, replicate put and update operations by copying the
     * relevant key and value across the wire (as opposed to just the key).
     */
    boolean isReplicateViaCopy();

    /**
     * @return true if this cache gathers statistics.
     */
    boolean isStatisticsEnabled();

    /**
     * Changes the state of statistics gathering.
     *
     * @param enabled The stats collection is enabled if true - disabled otherwise.
     * @since 2.8.4
     */
    void setStatistics(boolean enabled);

    /**
     * Get usage statistics for the cache.
     * <p>
     * The statistics are sorted by the cache statistics key {@link CacheStatisticsKey#getLabel() labels}. Providing
     * statistics is an optional feature and no assumptions should be made by the caller about which (if any) statistics
     * will be returned.
     *
     * @return a map of statistics keys to suppliers for their values, sorted lexically by the statistics key's label
     * @since v2.4.0
     */
    @Nonnull
    SortedMap<CacheStatisticsKey, Supplier<Long>> getStatistics();

    /**
     * Returns the collector for this cache.
     *
     * @return The {@link CacheCollector} for this cache.
     * @since 2.8.4
     */
    @Nullable
    CacheCollector getCacheCollector();

}
