package com.atlassian.braid;

/**
 * Extension allow Braid to extend a primary type by adding fields in an extension type to the primary type. The object
 * value for the extension type is fetched by querying a top level field in the extension type's schema source. The
 * value of a field in the primary type is passed as an argument when querying the extension type's schema source.
 */
public final class Extension {
    /**
     * The primary type being extended.
     */
    private final String type;

    /**
     * The name of the primary type's field used as the argument to query the extension type's schema source.
     */
    private final String on;

    /**
     * The information needed to query the extension type's schema source.
     */
    private final By by;

    public Extension(String type, String on, By by) {
        this.type = type;
        this.on = on;
        this.by = by;
    }

    public String getType() {
        return type;
    }

    public By getBy() {
        return by;
    }

    public String getOn() {
        return on;
    }

    public static final class By {
        /**
         * the namespace of the schema where to find the extension type
         */
        private final SchemaNamespace namespace;

        /**
         * The name of the extension type in the schema referenced by {@link #namespace}
         */
        private final String type;

        /**
         * The name of the top level query field to fetch extension objects
         */
        private final String query;

        /**
         * The name of the query argument used to fetch the extension objects, this will be filled with the value of the {@link Extension#on} field value.
         */
        private final String arg;

        public By(SchemaNamespace namespace, String type, String query, String arg) {
            this.namespace = namespace;
            this.type = type;
            this.query = query;
            this.arg = arg;
        }

        public SchemaNamespace getNamespace() {
            return namespace;
        }

        public String getType() {
            return type;
        }

        public String getQuery() {
            return query;
        }

        public String getArg() {
            return arg;
        }
    }
}
