package com.atlassian.braid.source.yaml;

import com.atlassian.braid.DefaultArgumentValueProvider;
import com.atlassian.braid.Extension;
import com.atlassian.braid.FieldRename;
import com.atlassian.braid.Link;
import com.atlassian.braid.LinkArgument;
import com.atlassian.braid.SchemaNamespace;
import com.atlassian.braid.TypeRename;
import com.atlassian.braid.document.DocumentMapperFactory;
import com.atlassian.braid.document.DocumentMappers;
import com.atlassian.braid.java.util.BraidMaps;
import com.atlassian.braid.java.util.BraidObjects;
import com.atlassian.braid.source.SchemaLoader;
import com.atlassian.braid.source.StringSchemaLoader;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;

public class YamlRemoteSchemaSourceBuilder {

    public static SchemaLoader buildSchemaLoader(Map<String, Object> m) {
        return new StringSchemaLoader(SchemaLoader.Type.IDL, (String) m.get("schema"));
    }

    public static SchemaNamespace buildSchemaNamespace(Map<String, Object> m) {
        return SchemaNamespace.of((String) m.get("namespace"));
    }

    public static List<FieldRename> buildQueryFieldRenames(Map<String, Object> m) {
        return Optional.of(BraidMaps.get(m, "queryFields")
                .orElse(BraidMaps.get(m, "topLevelFields")
                        .orElse(emptyList())))
                .map(YamlRemoteSchemaSourceBuilder::getFieldAliasesFromObject)
                .orElseThrow(IllegalStateException::new);
    }

    public static List<FieldRename> buildMutationAliases(Map<String, Object> m) {
        return Optional.of(BraidMaps.get(m, "mutationFields")
                .orElse(emptyList()))
                .map(YamlRemoteSchemaSourceBuilder::getFieldAliasesFromObject)
                .orElseThrow(IllegalStateException::new);
    }

    public static List<TypeRename> buildTypeRenames(Map<String, Object> m) {
        return BraidMaps.get(m, "typeAliases")
                .map(BraidObjects::<Map<String, String>>cast)
                .orElse(emptyMap())
                .entrySet()
                .stream()
                .map(entry -> TypeRename.from(entry.getKey(), entry.getValue()))
                .collect(toList());
    }

    public static List<Link> buildLinks(Map<String, Object> m) {
        final SchemaNamespace fromNamespace = SchemaNamespace.of(getOrThrow(m, "namespace"));

        return Stream.concat(buildSimpleLinks(fromNamespace, m), buildComplexLinks(fromNamespace, m))
                .collect(Collectors.toList());
    }

    private static Stream<Link> buildSimpleLinks(SchemaNamespace fromNamespace, Map<String, Object> m) {
        return getOrDefault(m, "links", Collections.<Map<String, Map<String, Object>>>emptyList())
                .stream()
                .map(link -> buildSimpleLink(fromNamespace, link));
    }

    private static Stream<Link> buildComplexLinks(SchemaNamespace fromNamespace, Map<String, Object> m) {
        return getOrDefault(m, "complexLinks", Collections.<Map<String, Object>>emptyList())
                .stream()
                .map(link -> buildComplexLink(fromNamespace, link));
    }

    public static List<Extension> buildExtensions(Map<String, Object> m) {
        return BraidMaps.get(m, "extensions")
                .map(BraidObjects::<List<Map<String, Object>>>cast)
                .map(YamlRemoteSchemaSourceBuilder::buildExtensions)
                .orElse(emptyList());
    }

    public static String buildOperationNamePrefix(Map<String, Object> m) {
        return getOrDefault(m, "operationNamePrefix", null);
    }

    private static List<Extension> buildExtensions(List<Map<String, Object>> extensions) {
        return extensions.stream().map(YamlRemoteSchemaSourceBuilder::buildExtension).collect(toList());
    }

    private static Extension buildExtension(Map<String, Object> e) {
        return new Extension(
                getOrThrow(e, "type"),
                getOrThrow(e, "field"),
                buildExtensionBy(getOrThrow(e, "by")));
    }

    private static Extension.By buildExtensionBy(Map<String, Object> by) {
        return new Extension.By(
                SchemaNamespace.of(getOrThrow(by, "namespace")),
                getOrThrow(by, "type"),
                getOrThrow(by, "query"),
                getOrThrow(by, "arg"));
    }

    public static DocumentMapperFactory buildDocumentMapperFactory(Map<String, Object> m) {
        return BraidMaps.get(m, "mapper")
                .map(BraidObjects::<List<Map<String, Object>>>cast)
                .map(DocumentMappers::fromYamlList)
                .orElse(DocumentMappers.identity());
    }

    private static List<FieldRename> getFieldAliasesFromObject(Object fields) {
        if (fields instanceof List) {
            return ((List<String>) fields).stream()
                    .map(f -> FieldRename.from(f, f))
                    .collect(toList());
        } else if (fields instanceof Map) {
            return ((Map<String, String>) fields).entrySet().stream()
                    .map(entry -> FieldRename.from(entry.getKey(), entry.getValue()))
                    .collect(toList());
        } else {
            throw new IllegalArgumentException("Unexpected field type");
        }
    }

    /**
     * A simple link is defined as a list item of "links".
     *  - from:
     *      type: String!                               # Link#sourceType
     *      field: String!                              # Link#newFieldName
     *      fromField: String=valueOf(field)            # LinkArgument#sourceName
     *      replaceFromField: Boolean=false             # LinkArgument#removeInputField
     *    to:
     *      namespace: String!                          # Link#targetNamespace
     *      type: String!                               # Link#targetType
     *      targetNonNullable: Boolean=false            # Link#targetNonNullable
     *      field: String=valueOf(from.field)           # Link#topLevelQueryField
     *      variableField: String=valueOf(argument)     # LinkArgument#targetFieldMatchingQueryArgument
     *      argument: String="id"                       # LinkArgument#queryArgumentName
     *      nullable: Boolean=false                     # LinkArgument#nullable
     */
    private static Link buildSimpleLink(SchemaNamespace fromNamespace, Map<String, Map<String, Object>> linkMap) {

        final Map<String, String> from = getOrThrow(linkMap, "from");
        final Map<String, Object> to = getOrThrow(linkMap, "to");

        Link.LinkBuilder linkBuilder = buildFrom(fromNamespace, from);
        linkBuilder = buildTo(linkBuilder, to);

        return linkBuilder.build();
    }

    /**
     * A complex link is defined as a list item of "complexLinks".
     *  - sourceType: String!                           # Link#sourceType
     *    targetType: String!                           # Link#targetType
     *    targetNonNullable: Boolean=false              # Link#targetNonNullable
     *    targetNamespace: String!                      # Link#targetNamespace
     *    topLevelQueryField: String!                   # Link#topLevelQueryField
     *    field: String!                                # Link#newFieldName
     *    arguments: List=[]                            # Link#linkArguments
     *    - sourceName: String!                         # LinkArgument#sourceName
     *      argumentSource: ArgumentSource!             # LinkArgument#argumentSource
     *      queryArgumentName: String!                  # LinkArgument#queryArgumentName
     *      nullable: Boolean=true                      # LinkArgument#nullable
     *      removeInputField: Boolean=false             # LinkArgument#removeInputField
     *      targetFieldMatchingArgument: String=null    # LinkArugment#targetFieldMatchingArgument
     */
    private static Link buildComplexLink(SchemaNamespace fromNamespace, Map<String, Object> linkMap) {
        Link.ComplexLinkBuilder linkBuilder = Link.newComplexLink()
                .sourceNamespace(fromNamespace)
                .sourceType(getOrThrow(linkMap, "sourceType"))
                .targetType(getOrThrow(linkMap, "targetType"))
                .targetNonNullable(getOrDefault(linkMap, "targetNonNullable", false))
                .targetNamespace(SchemaNamespace.of(getOrThrow(linkMap, "targetNamespace")))
                .topLevelQueryField(getOrThrow(linkMap, "topLevelQueryField"))
                .newFieldName(getOrThrow(linkMap, "field"));

        getOrDefault(linkMap, "arguments", Collections.<Map<String, Object>>emptyList())
                .stream()
                .map(YamlRemoteSchemaSourceBuilder::buildLinkArgument)
                .forEach(linkBuilder::linkArgument);

        BraidMaps.get(linkMap, "staticValues")
                .map(BraidObjects::<Map<String, Object>>cast)
                .map(DefaultArgumentValueProvider::new)
                .ifPresent(linkBuilder::argumentValueProvider);

        return linkBuilder.build();
    }

    private static LinkArgument buildLinkArgument(Map<String, Object> argumentMap) {
        return LinkArgument.newLinkArgument()
                .nullable(getOrDefault(argumentMap, "nullable", true))
                .removeInputField(getOrDefault(argumentMap, "removeInputField", false))
                .sourceName(getOrThrow(argumentMap, "sourceName"))
                .targetFieldMatchingArgument(getOrDefault(argumentMap, "targetFieldMatchingArgument", null))
                .queryArgumentName(getOrThrow(argumentMap, "queryArgumentName"))
                .argumentSource(LinkArgument.ArgumentSource.valueOf(getOrThrow(argumentMap, "argumentSource")))
                .build();
    }

    private static Link.LinkBuilder buildFrom(SchemaNamespace fromNamespace, Map<String, String> from) {
        return Link.newSimpleLink()
                .sourceNamespace(fromNamespace)
                .sourceType(getOrThrow(from, "type"))
                .newFieldName(getOrThrow(from, "field"))
                .sourceInputFieldName(getOrThrow(from, "fromField", "field"))
                .removeInputField(getOrDefault(from, "replaceFromField", false));
    }

    private static Link.LinkBuilder buildTo(Link.LinkBuilder builder, Map<String, Object> to) {
        return builder
                .targetNamespace(SchemaNamespace.of(getOrThrow(to, "namespace")))
                .targetType(getOrThrow(to, "type"))
                .targetNonNullable(getOrDefault(to, "targetNonNullable", false))
                .topLevelQueryField(getOrDefault(to, "field", null))
                .targetFieldMatchingQueryArgument(getOrDefault(to, "variableField", null))
                .queryArgumentName(getOrDefault(to, "argument", null))
                .setNullable(getOrDefault(to, "nullable", false));
    }

    private static <T> T getOrThrow(Map<String, ?> map, String key) {
        return BraidMaps.get(map, key)
                .map(BraidObjects::<T>cast)
                .orElseThrow(IllegalStateException::new);
    }

    private static <T> T getOrDefault(Map<String, ?> map, String key, T defaultValue) {
        return BraidMaps.get(map, key)
                .map(BraidObjects::<T>cast)
                .orElse(defaultValue);
    }

    private static <T> T getOrThrow(Map<String, ?> map, String key, String alternativeKey) {
        return BraidMaps.get(map, key)
                .map(BraidObjects::<T>cast)
                .orElse(getOrThrow(map, alternativeKey));
    }
}
