package com.atlassian.braid;

import graphql.language.InputValueDefinition;
import graphql.language.ListType;
import graphql.language.NonNullType;
import graphql.language.ObjectTypeDefinition;
import graphql.language.Type;
import graphql.language.TypeDefinition;
import graphql.language.TypeName;
import graphql.schema.idl.TypeDefinitionRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.braid.TypeUtils.DEFAULT_QUERY_TYPE_NAME;
import static com.atlassian.braid.TypeUtils.findMutationType;
import static com.atlassian.braid.TypeUtils.findQueryType;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;

/**
 * This wraps a {@link SchemaSource} to enhance it with helper functions
 */
public final class BraidSchemaSource {
    private static final Logger log = LoggerFactory.getLogger(BraidSchemaSource.class);

    private final SchemaSource schemaSource;

    private final TypeDefinitionRegistry registry;
    private final ObjectTypeDefinition queryType;

    private final ObjectTypeDefinition mutationType;
    public BraidSchemaSource(SchemaSource schemaSource) {
        this.schemaSource = requireNonNull(schemaSource);
        this.registry = schemaSource.getSchema();
        this.queryType = findQueryType(registry).orElse(null);
        this.mutationType = findMutationType(registry).orElse(null);
    }

    public SchemaSource getSchemaSource() {
        return schemaSource;
    }

    SchemaNamespace getNamespace() {
        return schemaSource.getNamespace();
    }

    Optional<TypeAlias> getTypeAlias(String type) {
        return schemaSource.getTypeAliases().stream().filter(a -> a.getSourceName().equals(type)).findFirst();
    }

    public Optional<TypeAlias> getTypeAliasFromAlias(String type) {
        return schemaSource.getTypeAliases().stream().filter(a -> a.getBraidName().equals(type)).findFirst();
    }

    Optional<FieldAlias> getQueryFieldAlias(String type) {
        return getFieldAlias(schemaSource.getQueryFieldAliases(), type);
    }

    Optional<FieldAlias> getMutationFieldAliases(String type) {
        return getFieldAlias(schemaSource.getMutationFieldAliases(), type);
    }

    /**
     * Gets the actual source type, accounting for links to query objects that have been renamed when merged into the
     * Braid schema
     */
    public String getLinkBraidSourceType(Link link) {
        return getQueryType().map(originalQueryType -> {
            if (originalQueryType.getName().equals(link.getSourceType())) {
                return DEFAULT_QUERY_TYPE_NAME;
            } else {
                return link.getSourceType();
            }
        }).orElse(link.getSourceType());
    }

    Collection<BraidTypeDefinition> getNonOperationTypes() {
        return registry.types()
                .values()
                .stream()
                .filter(this::isNotOperationType)
                .map(td -> new BraidTypeDefinition(this, td))
                .collect(toList());
    }

    boolean hasType(String type) {
        return registry.getType(type).isPresent();
    }

    Optional<ObjectTypeDefinition> getQueryType() {
        return Optional.ofNullable(queryType);
    }

    Optional<ObjectTypeDefinition> getMutationType() {
        return Optional.ofNullable(mutationType);
    }

    TypeDefinitionRegistry getTypeRegistry() {
        return registry;
    }

    public Type aliasType(Type type) {
        if (type instanceof TypeName) {
            final String typeName = ((TypeName) type).getName();
            TypeAlias alias = getTypeAlias(typeName).orElse(TypeAlias.from(typeName, typeName));
            return new TypeName(alias.getBraidName());
        } else if (type instanceof NonNullType) {
            return new NonNullType(aliasType(((NonNullType) type).getType()));
        } else if (type instanceof ListType) {
            return new ListType(aliasType(((ListType)type).getType()));
        } else {
            // TODO handle all definition types (in a generic enough manner)
            log.error("Definition type : " + type + " not handled correctly for aliases.  Please raise an issue.");
            return type;
        }
    }

    public Type unaliasType(Type type) {
        if (type instanceof TypeName) {
            final String typeName = ((TypeName) type).getName();
            TypeAlias alias = getTypeAliasFromAlias(typeName).orElse(TypeAlias.from(typeName, typeName));
            return new TypeName(alias.getSourceName());
        } else if (type instanceof NonNullType) {
            return new NonNullType(unaliasType(((NonNullType) type).getType()));
        } else if (type instanceof ListType) {
            return new ListType(unaliasType(((ListType)type).getType()));
        } else {
            // TODO handle all definition types (in a generic enough manner)
            log.error("Definition type : " + type + " not handled correctly for aliases.  Please raise an issue.");
            return type;
        }
    }

    List<InputValueDefinition> aliasInputValueDefinitions(List<InputValueDefinition> inputValueDefinitions) {
        return inputValueDefinitions.stream()
                .map(input ->
                        new InputValueDefinition(
                                input.getName(),
                                aliasType(input.getType()),
                                input.getDefaultValue(),
                                input.getDirectives()))
                .collect(toList());
    }

    private Optional<FieldAlias> getFieldAlias(List<FieldAlias> aliases, String type) {
        // if no mappings explicit mappings defined then expose all fields
        if(aliases.isEmpty()) {
            return Optional.of(FieldAlias.from(type, type));
        }
        return aliases.stream().filter(a -> a.getSourceName().equals(type)).findFirst();
    }

    private boolean isNotOperationType(TypeDefinition typeDefinition) {
        return !isOperationType(typeDefinition);
    }

    private boolean isOperationType(TypeDefinition typeDefinition) {
        requireNonNull(typeDefinition);
        return Objects.equals(queryType, typeDefinition) || Objects.equals(mutationType, typeDefinition);
    }
}
