define('bitbucket/internal/widget/tree-view/tree-builder', ['exports', 'bitbucket/internal/widget/tree-view/enums'], function (exports, _enums) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.flattenTree = flattenTree;
    exports.buildTreeFromNodes = buildTreeFromNodes;
    exports.buildNodesRelation = buildNodesRelation;
    exports.buildTree = buildTree;

    function _objectWithoutProperties(obj, keys) {
        var target = {};

        for (var i in obj) {
            if (keys.indexOf(i) >= 0) continue;
            if (!Object.prototype.hasOwnProperty.call(obj, i)) continue;
            target[i] = obj[i];
        }

        return target;
    }

    var _extends = Object.assign || function (target) {
        for (var i = 1; i < arguments.length; i++) {
            var source = arguments[i];

            for (var key in source) {
                if (Object.prototype.hasOwnProperty.call(source, key)) {
                    target[key] = source[key];
                }
            }
        }

        return target;
    };

    var createRootNode = function createRootNode() {
        return {
            id: _enums.ROOT_ID,
            isRoot: true,
            children: {}
        };
    };

    var createParentNode = function createParentNode(_ref) {
        var id = _ref.id,
            name = _ref.name;
        return {
            id: id,
            name: name,
            type: _enums.NODE_TYPE_DIRECTORY,
            isCollapsed: false,
            children: {}
        };
    };

    var getParentId = function getParentId(name, index) {
        return name.slice(0, index + 1).join(_enums.PATH_SEPARATOR);
    };

    var getOrCreateNodeFromPath = function getOrCreateNodeFromPath(root, path) {
        return path.reduce(function (parent, name, index) {
            if (!parent.children[name]) {
                var parentId = getParentId(path, index);

                parent.children[name] = createParentNode({ id: parentId, name: name });
            }

            return parent.children[name];
        }, root);
    };

    /** @visibleForTesting */
    function flattenTree(tree) {
        var children = tree.children;


        var hasSingleChild = function hasSingleChild(node) {
            return node.children.length === 1;
        };
        var areAllChildrenADirectory = function areAllChildrenADirectory(node) {
            return node.children.every(function (childNode) {
                return childNode.type === _enums.NODE_TYPE_DIRECTORY;
            });
        };

        children.forEach(function (node, index) {
            var childNode = node;
            var path = [node.name];

            while (hasSingleChild(childNode) && areAllChildrenADirectory(childNode)) {
                childNode = childNode.children[0];

                path.push(childNode.name);
            }

            // If we didn't end up on the same tree level
            if (childNode !== node) {
                var newBranchName = path.join(_enums.PATH_SEPARATOR);

                children.splice(index, 1, _extends({}, childNode, {
                    id: childNode.id,
                    name: newBranchName
                }));

                flattenTree(childNode);
            } else if (childNode.type === _enums.NODE_TYPE_DIRECTORY) {
                flattenTree(childNode);
            }
        });
    }

    function transformChildrenToArray(node) {
        var children = Object.values(node.children);

        children.forEach(transformChildrenToArray);

        node.children = children;
    }

    function sortTreeNodes(node, sortComparator) {
        var children = node.children;


        children.sort(function (_ref2, _ref3) {
            var pathA = _ref2.name,
                typeA = _ref2.type;
            var pathB = _ref3.name,
                typeB = _ref3.type;
            return sortComparator(pathA, typeA, pathB, typeB);
        });

        children.forEach(function (node) {
            return sortTreeNodes(node, sortComparator);
        });
    }

    var getPathToLeaf = function getPathToLeaf(path) {
        return path.slice(0, -1);
    };

    /** @visibleForTesting */
    function buildTreeFromNodes(nodes) {
        var tree = nodes.reduce(function (root, _ref4) {
            var name = _ref4.name,
                path = _ref4.path,
                node = _objectWithoutProperties(_ref4, ['name', 'path']);

            var pathToLeaf = getPathToLeaf(path);

            var _getOrCreateNodeFromP = getOrCreateNodeFromPath(root, pathToLeaf),
                children = _getOrCreateNodeFromP.children;

            children[name] = _extends({}, node, {
                name: name,
                isCollapsed: false,
                children: {}
            });

            return root;
        }, createRootNode());

        transformChildrenToArray(tree);

        return tree;
    }

    function buildNodesRelation(tree) {
        var parentsRelation = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
        var children = tree.children,
            parentId = tree.id;


        children.forEach(function (child) {
            parentsRelation[child.id] = parentId;

            buildNodesRelation(child, parentsRelation);
        });

        return parentsRelation;
    }

    function buildTree(nodes, sortComparator) {
        var tree = buildTreeFromNodes(nodes);

        flattenTree(tree);
        sortTreeNodes(tree, sortComparator);

        return tree;
    }
});