define('bitbucket/internal/util/store/selectors', ['exports', 'lodash', 'reselect', './reducers'], function (exports, _lodash, _reselect, _reducers) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports._defaultTransform = exports.getSelectorsForApp = exports.keyPathForProps = exports.singlePropSelector = exports.filteredPropsSelector = undefined;

    function _toConsumableArray(arr) {
        if (Array.isArray(arr)) {
            for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) {
                arr2[i] = arr[i];
            }

            return arr2;
        } else {
            return Array.from(arr);
        }
    }

    /**
     * Build a props object with key names the same as those in filteredProps
     * @param {array<string>} filteredProps - The prop names to filter out
     * @returns {Function}
     */
    var defaultTransform = function defaultTransform(filteredProps) {
        return function () {
            for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
                args[_key] = arguments[_key];
            }

            return filteredProps.reduce(function (out, prop, i) {
                out[prop] = args[i];
                return out;
            }, {});
        };
    };

    /**
     * A selector the picks a subset of the props and optionally transforms them.
     * Ignores state.
     *
     * @param {array<string>} filteredProps - The prop names to filter out
     * @param {function?} transform - An optional function that takes each prop as a separate argument,
     *                                  in the order specified in `filteredProps`
     * @returns {function(*, *=): *}
     */
    var filteredPropsSelector = exports.filteredPropsSelector = function filteredPropsSelector(filteredProps) {
        var transform = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : defaultTransform(filteredProps);

        var selector = (0, _reselect.createSelector)(filteredProps.map(function (prop) {
            return function (props) {
                return props[prop];
            };
        }), transform);

        //Need to discard state below and not inside the generated prop selectors above in order
        //to avoid incorrectly invalidating the selector when the state changes but the props don't
        return function (state, props) {
            return selector(props);
        };
    };

    /**
     * Convenience wrapper around `filteredPropsSelector` for when you just want a single prop returned directly
     * (not inside a props object)
     * @param {string} prop - The prop to return
     * @returns {function(*, *=): *}
     */
    var singlePropSelector = exports.singlePropSelector = function singlePropSelector(prop) {
        return filteredPropsSelector([prop], function (a) {
            return a;
        });
    };

    /**
     * Build the path string for a key hierarchy generated by util/reducers `keyByRequestProps`
     *
     * @param {string[]} props - the names of the properties to use
     * @param {object} payload
     *
     * @returns {*}
     */
    var keyPathForProps = exports.keyPathForProps = function keyPathForProps(props, payload) {
        return props.map(function (prop) {
            return (0, _reducers.getKeyFromProp)(prop, payload);
        });
    };

    /**
     * Build the path in the entity/sub-app state that matches the shape generated by
     * util/reducers `singleEntityRequestReducer`/`getRequestPageReducersByType`
     * @param {object} requestTypeKeyProps - The map of request types to the payload properties used to key the request
     * @param {function} getId - the id getter for this entity type.
     *
     * @returns {function(*): array}
     */
    var getRequestPathBuilder = function getRequestPathBuilder() {
        var requestTypeKeyProps = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        var getId = arguments[1];
        var requestKeyCreator = arguments[2];
        return function (action) {
            var payload = action.payload,
                type = action.type;
            var requestType = payload.requestType;


            if (requestKeyCreator) {
                var requestPaths = requestKeyCreator(action);

                return ['requests'].concat(_toConsumableArray(requestPaths));
            }

            var commonPath = ['requests', type];

            if (requestType) {
                return [].concat(commonPath, [requestType], _toConsumableArray(keyPathForProps(requestTypeKeyProps[requestType], payload)));
            }

            // For single entity requests, assume the results are keyed by the payload id
            // (see util/reducers:keyByPayloadId)
            return [].concat(commonPath, [getId(payload)]);
        };
    };

    /**
     * Build the collection of standard selectors for an app.
     * Pass the same value for `requestTypeKeyProps` as that used for util/reducers `getRequestPageReducersByType`
     * to build a matching request selector.
     *
     * @param {string} Entity - the entity type (see bitbucket/internal/enums::Entity)
     * @param {function} getId - the id getter for this entity type.
     * @param {object} requestTypeKeyProps - The map of request types to the payload properties used to key the request
     *
     * @returns {object}
     */

    var getSelectorsForApp = exports.getSelectorsForApp = function getSelectorsForApp(_ref) {
        var appName = _ref.appName,
            getId = _ref.getId,
            requestTypeKeyProps = _ref.requestTypeKeyProps,
            requestKeyCreator = _ref.requestKeyCreator;

        var requestPathBuilder = getRequestPathBuilder(requestTypeKeyProps, getId, requestKeyCreator);

        var getAppState = function getAppState(state) {
            return (0, _lodash.get)(state, appName);
        };

        var getEntities = (0, _reselect.createSelector)([getAppState], function (state) {
            return (0, _lodash.get)(state, 'entities');
        });

        var getEntity = function getEntity(minimalEntitySelector) {
            return (0, _reselect.createSelector)([getEntities, minimalEntitySelector], function (entities, minimalEntity) {
                return (0, _lodash.get)(entities, getId(minimalEntity));
            });
        };

        var getRequestInfo = function getRequestInfo(actionSelector) {
            return (0, _reselect.createSelector)([getAppState, actionSelector], function (state, action) {
                return (0, _lodash.get)(state, requestPathBuilder(action));
            });
        };

        var getEntitiesForRequest = function getEntitiesForRequest(actionSelector) {
            return (0, _reselect.createSelector)([getRequestInfo(actionSelector), getEntities], function (request, entities) {
                var requestIds = (0, _lodash.get)(request, 'ids');

                return requestIds && requestIds.length ? requestIds.map(function (id) {
                    return entities[id];
                }) : _reducers.EMPTY_ARRAY;
            });
        };

        var getLoadingForRequest = function getLoadingForRequest(actionSelector) {
            return (0, _reselect.createSelector)([getRequestInfo(actionSelector)], function (requestInfo) {
                return (0, _lodash.get)(requestInfo, 'loading', false);
            });
        };

        var getPagingMetaForRequest = function getPagingMetaForRequest(actionSelector) {
            return (0, _reselect.createSelector)([getRequestInfo(actionSelector)], function (requestInfo) {
                return (0, _lodash.get)(requestInfo, 'pagingMeta', {});
            });
        };

        var getResponseForRequest = function getResponseForRequest(actionSelector) {
            return (0, _reselect.createSelector)([getRequestInfo(actionSelector)], function (requestInfo) {
                return (0, _lodash.get)(requestInfo, 'response', {});
            });
        };

        var getAllFetchedForRequest = function getAllFetchedForRequest(actionSelector) {
            return (0, _reselect.createSelector)([getPagingMetaForRequest(actionSelector)], function (requestInfo) {
                return (0, _lodash.get)(requestInfo, 'isLastPage', false);
            });
        };

        return {
            getAllFetchedForRequest: getAllFetchedForRequest,
            getAppState: getAppState,
            getEntities: getEntities,
            getEntitiesForRequest: getEntitiesForRequest,
            getEntity: getEntity,
            getLoadingForRequest: getLoadingForRequest,
            getPagingMetaForRequest: getPagingMetaForRequest,
            getResponseForRequest: getResponseForRequest,
            getRequestInfo: getRequestInfo
        };
    };

    //Expose for testing only
    exports._defaultTransform = defaultTransform;
});