define('bitbucket/internal/util/svg', ['module', 'exports', 'lodash'], function (module, exports, _lodash) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });

    var _lodash2 = babelHelpers.interopRequireDefault(_lodash);

    var svgNS = 'http://www.w3.org/2000/svg';

    /**
     * Create an SVG element using document.createElementNS('http://www.w3.org/2000/svg');
     *
     * @param {string} name - element name to create
     * @param {Object} attrs - map of attribute kye to value to set on the element
     * @returns {Element}
     */
    function createElement(name, attrs) {
        var el = document.createElementNS(svgNS, name);
        _lodash2.default.forEach(attrs || [], function (attrValue, attrKey) {
            el.setAttribute(attrKey, attrValue);
        });
        return el;
    }

    /**
     * Used for generating a path string you can use in <path d="{here}">
     * @constructor
     */
    function PathBuilder() {
        this._buffer = [];
    }

    /**
     * Move to a point
     *
     * @param {number} x
     * @param {number} y
     * @returns {PathBuilder}
     */
    PathBuilder.prototype.moveTo = function (x, y) {
        this._buffer.push('M', x, y);
        return this;
    };

    /**
     * Draw a Bezier curve through two control points to an end point
     *
     * @param {number} x1 - the first control point x-value
     * @param {number} y1 - the first control point y-value
     * @param {number} x2 - the second control point x-value
     * @param {number} y2 - the second control point y-value
     * @param {number} x3 - the end point x-value
     * @param {number} y3 - the end point y-value
     * @returns {PathBuilder}
     */
    PathBuilder.prototype.curve = function (x1, y1, x2, y2, x3, y3) {
        this._buffer.push('C', x1, y1, x2, y2, x3, y3);
        return this;
    };

    /**
     * Draw a line to a point
     *
     * @param {number} x
     * @param {number} y
     * @returns {PathBuilder}
     */
    PathBuilder.prototype.lineTo = function (x, y) {
        this._buffer.push('L', x, y);
        return this;
    };

    /**
     * Return to the start of the path
     * @returns {PathBuilder}
     */
    PathBuilder.prototype.close = function () {
        this._buffer.push('Z');
        return this;
    };

    /**
     * Return the path string generated by this builder.
     * @returns {string}
     */
    PathBuilder.prototype.build = function () {
        return this._buffer.join(' ');
    };

    exports.default = {
        svgNS: svgNS,
        createElement: createElement,
        PathBuilder: PathBuilder
    };
    module.exports = exports['default'];
});